// -*- c++ -*-
/*
-------------------------------------------------------------------------
This file is part of WxWidgetsExtensions library.
-------------------------------------------------------------------------

WxExtLib (WxWidgetsExtensions) library
-----------------------------

COPYRIGHT NOTICE:

WxExtLib library Copyright (c) 2003-2007 Daniel Kps

The WxWidgetsExtensions library and associated documentation files (the
"Software") is provided "AS IS".  The author(s) disclaim all
warranties, expressed or implied, including, without limitation, the
warranties of merchantability and of fitness for any purpose.  The
author(s) assume no liability for direct, indirect, incidental,
special, exemplary, or consequential damages, which may result from
the use of or other dealings in the Software, even if advised of the
possibility of such damage.

Permission is hereby granted, free of charge, to any person obtaining
a copy of this Software, to deal in the Software without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Software,
and to permit persons to whom the Software is furnished to do so,
subject to the following conditions:

 1. The origin of this source code must not be misrepresented.
 2. Altered versions must be plainly marked as such and must not be
    misrepresented as being the original source.
 3. This Copyright notice may not be removed or altered from any 
    source or altered source distribution.

End of WxExtLib library Copyright Notice

-------------------------------------------------------------------------
*/

#ifndef _INCLUDED_WxMisc_h
#define _INCLUDED_WxMisc_h

#if defined(__GNUG__) && (!defined(__APPLE__)) && (!defined(M_NoPragmaInterface))
#   pragma interface "WxMisc.h"
#endif

#include "WxExtLibConfig.h"

#include <wx/defs.h>

#ifndef WX_PRECOMP
#   include <wx/longlong.h>
#   if wxUSE_GUI
#       include <wx/gdicmn.h>
#   endif
#   include <wx/dynarray.h>
#   if wxUSE_GUI
#       include <wx/event.h>
// HACK to avoid error "base class `wxWindow' has incomplete type": include
// something which includes the necessary header files
#       include <wx/stattext.h>
#   endif
#ifndef M_MinimalTimeInterval_UseGetTickCount
#   include <wx/timer.h>
#endif
#endif

#if wxUSE_GUI
#   include <wx/image.h>
#endif

#include <wx/datetime.h>

#if wxUSE_GUI
#   include <wx/html/htmltag.h>
#   include <wx/html/htmlwin.h>
#   include <wx/textctrl.h>

#   include <wx/brush.h> // for class definition of wxBrush
#   include <wx/dcclient.h>
#endif // wxUSE_GUI

#if defined(__WXMSW__)
#   include <windows.h> // WARN may not be the right include place here
// #   include <wx/msw/dibutils.h> // commented out for wxWindows 2.5.2

#if wxUSE_GUI
// NOTE
// for wxWindows 2.5.2, dibutils.h disappeared so we provide an own
// HDIB definition (now called WXHDIB for consistency with WXHWND etc.)
typedef HGLOBAL WXHDIB;
#endif // wxUSE_GUI

#endif

// wxMessageFormatter/wxFormat/wxFormatMessage has been moved into
// extra header and source file
#include "WxMessageFormat.h"

//-------------------------------------------------------------------------

#ifndef WX_DEFINE_ARRAY_PTR
// under wxWindows 2.5.2, WX_DEFINE_ARRAY_PTR should be used, but it
// doesn't seem to be available under wxWindows 2.4.1
#   define WX_DEFINE_ARRAY_PTR WX_DEFINE_ARRAY
#endif

//-------------------------------------------------------------------------

WX_DEFINE_ARRAY_PTR(wxObject *, wxObjectPtrArray);

WX_DEFINE_ARRAY_PTR(wxWindow *, wxWindowPtrArray);

//-------------------------------------------------------------------------

#ifndef for_once
#define for_once  for (int _for_once___ ## __LINE__=0; _for_once___ ## __LINE__ < 1; ++ _for_once___ ## __LINE__)
#endif

//-------------------------------------------------------------------------

#define M_IsDefineExportMagic 1
#    include "WxExtLibAliases.h"
#undef M_IsDefineExportMagic

//-------------------------------------------------------------------------
// wxMinimalTimeInterval
//
// Alternative implementations - and their problems (scope wxWidgets 2.4):
// 0) wxStopWatch - timer overflow (calls wxLongLong.GetLo(), 
//    uses wxGetLocalTimeMillis(), same problems as with wxGetLocalTimeMillis())
// 1) wxGetLocalTimeMillis() - problem after adjusting the system clock (e.g. under Win98SE),
//    but quasi no timer overflow (wxLongLong is used)
// 2) ::GetTickCount() (Win32) - timer overflow after about 48 hours
// 3) ::timeGetTime() (Win32, winmm.lib) - timer overflow after about 48 hours
//
// Now done above where headers are included:
// #ifdef __WIN32__
// #    define M_MinimalTimeInterval_UseGetTickCount
// #endif

class wxMinimalTimeInterval
{
 public:
    wxMinimalTimeInterval ();
    
    void setMinimalTimeInterval (long MinimalTimeIntervalMS);

    void setBegin ();
    bool check ();

 private:
#ifdef M_MinimalTimeInterval_UseGetTickCount
    DWORD m_BeginUpTimeMS;
#else
    // wxStopWatch m_StopWatch;
    wxLongLong m_BeginTimeMSLongLong;
#endif
    long m_MinimalTimeIntervalMS;
    bool m_IsInitialized;
};

//-------------------------------------------------------------------------

class wxScopedSetToTrue
{
 public:
    wxScopedSetToTrue (bool & IsLockedRef)
      : m_IsLockedRef (IsLockedRef)
      {
        m_PreviousIsLocked = IsLockedRef;
        m_IsLockedRef = true;
      }
    ~wxScopedSetToTrue ()
      {
        m_IsLockedRef = m_PreviousIsLocked;
      }
 private:
    bool & m_IsLockedRef;
    bool m_PreviousIsLocked;
};

//-------------------------------------------------------------------------

extern bool convertStringToLong (const char * CharPtr, long & Long);
extern bool convertStringToLong (const char * CharPtr, unsigned long & Long);

//-------------------------------------------------------------------------

// convertAsciiToHtml(): 
// - useful when generating HTML code: will escape the
//   characters '<', '>', '"' (quotation mark) and '&' to their 
//   HTML entities '&lt;', '&gt;', '&quot;' and '&amp;', respectively
//   (If this escaping isn't done, syntactically incorrect HTML code may
//   result).
//
// - NOTE translation of ISO 8859-1/Latin-1 specific characters (128-255) 
//   (e.g. umlauts, accented characters etc.) is not needed if we specify 
//   the ISO Latin-1 encoding/charset in the HTML header - example:
//   <head>
//     ...
//     <meta http-equiv="content-type" content="text/html; charset=iso-8559-1">
//     ...
//   </head>
//   If no charset was specified, possibly the (system) default charset is used.
//   Translation of characters (128-255) (for all different charsets) remains 
//   as a to-do, however.
//
// - EWxAsciiToHtmlFlags:
//   - ConvertLeadingSpaces: replace leading space by "&nbsp;" (tab-char is treated as four spaces)
//   - ConvertAllSpaces: replace any space by "&nbsp;" (tab-char is treated as four spaces)
// 
// - IsTranslateNewlines: replace newline char (LF) with "<br>" (HTML line break tag)
enum EWxAsciiToHtmlFlags
  {
    wxAsciiToHtml_ConvertLeadingSpaces = 0x0001,
    wxAsciiToHtml_ConvertAllSpaces = 0x0002
  };

extern void convertAsciiToHtml (const char * AsciiCharPtr, wxString & HtmlString,
                                bool IsTranslateNewlines = false,
                                long WxAsciiToHtmlFlags = 0);
extern wxString convertAsciiToHtml (const char * AsciiStringCharPtr,
                                    bool IsTranslateNewlines = false,
                                    long WxAsciiToHtmlFlags = 0);

extern wxString formatColourToHtmlParam (const wxColour & Colour);

//-------------------------------------------------------------------------

// get double parameter independent of the locale setting(for restrictions, see
// comments in the implementation of parseDoubleFromAnyLocaleString())
extern bool getHtmlTagParamAsDouble (const wxHtmlTag & HtmlTag,
                                     const wxString & ParamNameString, double * Double);

// parseDoubleFromAnyLocaleString():
// - get double from a string in current locale or C locale (this is possibly
//   useful if both variants were mixed for whatever reason)
// Restrictions:
// - currently only accepts ',' or '.' as decimal point separator character
// - currently no thousand-separator is supported
// - for details, see comments in the implementation of the function
// TODO:
// - make a test print of known double value to determine the decimal point
//   character for the current locale
// - rename the function - the name parseDoubleFromAnyLocaleString() is wrong
extern bool parseDoubleFromAnyLocaleString (const wxString & InputValueString, double * Double);

//-------------------------------------------------------------------------

// locale-dependent number formatting/parsing general problems:
// - string 2.345 is not the same number in different locales: in English 
//   this is a 2 and 0.345 fractional part, while in German and some other 
//   European countries it could be 2345 (with grouping character point to 
//   separate the thousand part)
//   - likewise, interpretation of 2,345 depends on locales
// - used scanf() may not interpret what printf() has printed (e.g. if
//   scanf() and printf() use different libraries interpreting locale settings 
//   differently)
// - there seems to be no (generally available) OS or standard library function
//   to determine what the used grouping and decimal point characters are
//   - this is solved by using test data, which works at least for some or 
//     most western country locales
// - under certain circumstances it is desireable to force use of a certain
//   notation, even if user has set up environment variables or configured the
//   system to use a specific notation
//   - using environment variables to switch formatting behavior seems to be 
//     clumsy (if this works at all) and is probably not thread-safe

// wxNumberFormatOptions
// - contain desired number format
class wxNumberFormatOptions
{
public:
    wxNumberFormatOptions ();
    wxNumberFormatOptions (int GroupSize, char GroupSeparatorChar, 
                           char DecimalSeparatorChar);
    void set (int GroupSize, char GroupSeparatorChar, char DecimalSeparatorChar);

    int m_GroupSize;
    char m_GroupSeparatorChar;
    char m_DecimalSeparatorChar;
};

//-------------------------------------------------------------------------

// wxNumberFormattingInternals:
// - contain information internally used
// - to avoid sample formatting for each call to formatDouble() etc.,
//   a pointer to an object of this class may be given as argument to
//   formatDouble()
class wxNumberFormattingInternals
{
 public:
    wxNumberFormattingInternals();

    bool m_IsOkay;
    // values used by the standard-library printf() function
    char m_DecimalSeparatorChar;
};

//-------------------------------------------------------------------------

// detectNumberFormattingInternals():
// - currently can only detect either point ('.') or comma (',') as
//   decimal separator characters
extern void detectNumberFormattingInternals(wxNumberFormattingInternals * NumberFormattingInternals);

// formatLong()/formatDouble()/formatNumberStringWithGrouping():
// todo:
// - HTML formatting (e.g. exponential display with superscript)
// - alignment to decimal separator is changed because of inserted grouping
//   characters
// - split off into length determination and string modification function
//   and make it work with simple C-strings to be able to avoid heap memory 
//   usage of wxString for applications where performance is important
// - NumberFormattingInternals() could be made a thread-local variable
// - add support for grouping with hexadecimal digits (also see C99 %a 
//   floating-point hexadecimal conversion)
// - add parseDouble(), parseLong() that can deal with grouping characters

extern void formatLong (const wxString & FormatString, 
                        wxNumberFormatOptions & NumberFormatOptions, 
                        long Long, 
                        wxString & String, 
                        wxNumberFormattingInternals * NumberFormattingInternals = NULL);

extern void formatDouble (const wxString & FormatString, 
                          wxNumberFormatOptions & NumberFormatOptions, 
                          double Double, 
                          wxString & String, 
                          wxNumberFormattingInternals * NumberFormattingInternals = NULL);

// formatNumberStringWithGrouping()
// - insertion of grouping characters and decimal separator replacement is
//   only done on the first detected sequence of digits,
//   e.g. "123456.78 789012.34" -> "123.456,78 789012.34"
extern void formatNumberStringWithGrouping (const wxString & NumberString, 
                                            wxNumberFormatOptions & NumberFormatOptions, 
                                            wxString & String, 
                                            wxNumberFormattingInternals * NumberFormattingInternals);

// test function checking a few sample values to see if formatting
// works as expected
extern bool testNumberFormat();

//-------------------------------------------------------------------------

// DEF MonthIndex starts from 1, _not_ 0!
extern wxDateTime::Month getMonthFromMonthIndex (int MonthIndex);
// DEF returned month index starts from 1, _not_ 0!
extern int getMonthIndexFromMonth (wxDateTime::Month Month);

//-------------------------------------------------------------------------

// getDecomposedDateFormatString()
// - return decomposed format string according to current locale setting
// - example: when called with "%#x", the result may be "%a, %d %B %Y"
// implementation notes:
// - this uses a sample date to infer the decomposed format string
// - (there is no standard or OS specific function for this as it seems)
extern void getDecomposedDateFormatString (const wxString & InputFormatString,
                                           wxString & DecomposedDateFormatString);

//-------------------------------------------------------------------------

// formatDate()
// TODO
// - integrate into wxDateTime
extern void formatDate (const wxDateTime & DateTime, wxString & String,
                        bool IsLongFormat = true);

/*
 use the following fix for a bug in wxDateTime::GetWeekDayName():
    // FIX (probably all wxWindows versions, problem verified with 2.4.1, 2.5.2)
    // tm.tm_mday = 28; is not good because for wday >= Wed
    // the day will become invalid (e.g. 31.11.99) and CallStrftime
    // won't return the right thing - so use 21 (seven days before)
    // instead
    tm.tm_mday = 21;
    tm.tm_mon = Nov;
    tm.tm_year = 99;
*/

// parseDate()
// - will try to parse date in short, long (alternative) and ISO 8601 date format 
// Notes
// - parsing in short date format is tried with 2-digit and 4-digit year
// - parsing in long date format is tried with and without weekday name
// - therefore, five different format strings are tried
// - the short and long (alternative) date format are according the 
//   the current locale setting
// possible improvements:
// - allow abbreviations for month name and weekday name (could also
//   add a flag to allow abbreviations to wxDateTime class)
// - allow 2-digit years in long/alternative format
// - allow spaces to be used (implement in wxDateTime::ParseFormat())
// - check for consistency of weekday with date (weekday
//   is parsed but ignored) (fix it in wxDateTime::ParseFormat())
// TODO
// - integrate into wxDateTime
extern bool parseDate (const wxString & String, wxDateTime & DateTime);

// TODO:
// add support for time and date-time formatting and parsing

//-------------------------------------------------------------------------

// parse and format according to format string
// - DateTimeFormatString placeholders: see strftime() description
// - the parse function is very strict and may not allow additional spaces
extern bool parseDateTimeWithFormat (const wxString & InputString, wxDateTime & DateTime, 
                                     const wxString & DateTimeFormatString);
extern void formatDateTimeWithFormat (const wxDateTime & DateTime, wxString & String,
                                      const wxString & DateTimeFormatString);

//-------------------------------------------------------------------------

// parse and format in simple ISO format (e.g. 2005-05-14 17:05:34)
// - the parse function is very strict and may not allow additional spaces
extern bool parseDateTimeFromISOFormat (const wxString & InputString, wxDateTime & DateTime);
extern void formatDateTimeInISOFormat (const wxDateTime & DateTime, wxString & String);

//-------------------------------------------------------------------------

// fits object with ObjectSize extensions into FrameRect and returns
// result into FittedObjectRect. AlignmentFlags can be a combination of
// wxALIGN_CENTER_HORIZONTAL and wxALIGN_CENTER_VERTICAL
extern void fitInside (const wxRect & FrameRect, const wxSize & ObjectSize, 
                       wxRect & FittedObjectRect, int AlignmentFlags);

//-------------------------------------------------------------------------

// makePath()
// - concatenate LhsPathString and RhsPathString
// - RhsPathString must be a relative path
// - LhsPathString may have to be an absolute path (currently)
//   (if so, this should be extendend to non-absolute paths)
// - Example:
//     wxString Path = makePath (InstallPrefix, "data/file.dat");
extern void makePath (const wxString & LhsPathString,
                      const wxString & RhsPathString,
                      wxString & ResultPathString,
                      wxPathFormat LhsPathFormat = wxPATH_NATIVE,
                      wxPathFormat RhsPathFormat = wxPATH_UNIX,
                      wxPathFormat ResultPathFormat = wxPATH_NATIVE);
extern wxString makePath (const wxString & LhsPathString,
                          const wxString & RhsPathString,
                          wxPathFormat LhsPathFormat = wxPATH_NATIVE,
                          wxPathFormat RhsPathFormat = wxPATH_UNIX,
                          wxPathFormat ResultPathFormat = wxPATH_NATIVE);

extern void makeAbsolutePath (const wxString & RelativeOrAbsolutePathString,
                              const wxString & ReferenceDirString,
                              wxString & AbsolutePathString,
                              wxPathFormat RelativeOrAbsolutePathFormat = wxPATH_NATIVE,
                              wxPathFormat ReferenceDirFormat = wxPATH_NATIVE,
                              wxPathFormat AbsolutePathFormat = wxPATH_NATIVE);

extern wxString makeAbsolutePath (const wxString & RelativeOrAbsolutePathString,
                                  const wxString & ReferenceDirString,
                                  wxPathFormat RelativeOrAbsolutePathFormat = wxPATH_NATIVE,
                                  wxPathFormat ReferenceDirFormat = wxPATH_NATIVE,
                                  wxPathFormat AbsolutePathFormat = wxPATH_NATIVE);

extern void makeRelativePath (const wxString & RelativeOrAbsolutePathString,
                              const wxString & ReferenceDirString,
                              wxString & RelativePathString,
                              bool IsUseRelativeParentDir = true,
                              wxPathFormat RelativeOrAbsolutePathFormat = wxPATH_NATIVE,
                              wxPathFormat ReferenceDirFormat = wxPATH_NATIVE,
                              wxPathFormat AbsolutePathFormat = wxPATH_NATIVE);

extern wxString makeRelativePath (const wxString & RelativeOrAbsolutePathString,
                                  const wxString & ReferenceDirString,
                                  bool IsUseRelativeParentDir = true,
                                  wxPathFormat RelativeOrAbsolutePathFormat = wxPATH_NATIVE,
                                  wxPathFormat ReferenceDirFormat = wxPATH_NATIVE,
                                  wxPathFormat RelativePathFormat = wxPATH_NATIVE);

//-------------------------------------------------------------------------

// addConcatenatedStrings()
// - concatenates strings from given wxArrayString with specified 
//   item prefix, item postfix, between-item infix strings
// - adds elements in StringArray to OutputString, i.e. OutputString
//   is not deleted before
extern void addConcatenatedStrings (const wxArrayString & StringArray, 
                                    wxString & OutputString,
                                    const wxString & ElementPrefix = wxEmptyString,
                                    const wxString & ElementPostfix = wxEmptyString,
                                    const wxString & Infix = wxEmptyString);

//-------------------------------------------------------------------------

extern const char * wxGetIdTranslation(const char * MessageIdentCharPtr, const char * DefaultCharPtr);

//-------------------------------------------------------------------------

// wxGetCatalogMetaData()
// - return NULL if key is not found, otherwise returns meta-data value
//   (translation) 
// - syntax for meta-data keys could be e.g. (so it can be distinguished
//   from normal keys/translations)
//     {meta:Scope:Key}
const char * wxGetCatalogMetaData(const char * MessageIdentCharPtr);

//-------------------------------------------------------------------------

class wxLocale;

// wxCatalogAddManag
// - generate appropriate error messages if loading one or more catalogs
//   fails
// - calls to addCatalog(), addCatalogWithVersionCheck() are ignored
//   if associated locale doesn't have a language set (or used language
//   could not be determined)
// - reduce chances of forgotten catalog updates
// - simple support for catalog-version-numbers:
//   - avoid accidental use of outdated catalogs by a new program version,
//     which is important, because "wrong" printf() format strings may
//     cause program crashs
// - meta data version entry syntax: {meta:CATALOG-NAME:VERSION-IDENT}
//   - example: for a call to
//       addCatalogWithVersionCheck ("WxExtLib", "0.82.0003");
//     the following entry is expected:
//       {meta:WxExtLib:VersionIdent} having value 0.82.0003
// - warning: 
//   With the used meta-data version entry scheme, a wrong catalog 
//   version can be detected only after the catalog has been loaded,
//   so possibly the wrong catalog is used nevertheless (and there 
//   seems to be no way to remove a catalog from memory again). 
//   For certains applications it may be preferable to terminate 
//   the program soon in case of catalog version mismatch.
class wxCatalogAddManag
{
 public:
    wxCatalogAddManag();
    void init();

    // set locale to be used for calling AddCatalog()
    void setLocale (wxLocale * Locale);

    // addCatalog(), addCatalogWithVersionCheck()
    // - collects/appends error messages to m_MessageString, which
    //   later can be obtained by calling getMessage()
    // - the request to add a catalog is ignored (with no error)
    //   if no language is set for the associated locale (or if
    //   used language could not be determined)
    bool addCatalog (const wxString & CatalogName);
    bool addCatalogWithVersionCheck (const wxString & CatalogName,
                                     const wxString & RequiredVersionString);

    // getIsError()
    // - indicate if any catalog could not be loaded with specified
    //   version (or any version, if version is unspecified)
    bool getIsError();

    // getMessage()
    // - message returns language name used for specified locale
    //   and accepted vs. encountered catalog version numbers
    bool getMessage (wxString & MessageString);
    
 protected:
    wxLocale * m_Locale;
    bool m_IsError;
    wxString m_MessageString;
};

#if 0
extern bool checkCatalogueMetaData (wxString & MessageString,
                                    const char * CatalogueName,
                                    const char * VersionMessageIdentCharPtr,
                                    const char * RequiredVersionString);
#endif

//-------------------------------------------------------------------------

// string_local_language(), string_internal_language(), string_no_localization():
// notes on use of macro alias vs. inline function definition:
// - it is not fully clear if should be declared as macro or as function,
// - theoretically, use of function declaration instead of macro would seem
//   to be more safe somehow
// - however, use of macro would have the advantage of being able to #undef
//   it temporarily if not desired or gives conflicts
// - prefer using #define here, because _<single-char> is often defined as
//   macro (e.g. _L in MSVC to add Unicode prefix "L" to string literal)
// - with inline-function declaration, conflicts cannot be detected 
//   easily (typically confusing error message)
// - with macro definition, compiler may report a meaningful warning (at least
//   with a high warning level, or if enable specifically)

// string_local_language()
// - tag strings as being in 'local' language already
// - helps declaring strings which are already in local language 
//   for parts of a program/source code which doesn't need to support
//   messages in more than one language
// - _L() could be useful for an automated move to message catalogue
//   if internationalization is desired at some point in the future
// - comment: sometimes the simple things are the most useful ;)
// - implementation notes:
//   - would like to use _L(), but MSVC uses this to make argument a Unicode
//     string, so use _R (for Regionalized)
//   - don't use functional macro _R(Arg1), because _<single-char> is also used
//     as parameter-name in many headers of MSVC 6 STL header files
//       #define _R(Arg1) string_local_language(R)
//   - alternative would be to do
//       inline const char * _R(const char * CharPtr) { return CharPtr; }
//     but see above

inline const char * string_local_language(const char * CharPtr) { return CharPtr; }

#ifdef _R 
#   error "cannot define alias _R() for function string_local_language(), because _R is used as macro already"
#endif
// #define _R string_local_language
inline const char * _R(const char * CharPtr) { return CharPtr; }

// string_internal_language()
// - tag strings as being in internal default (normally English) 
//   language, but no translation is desired
// - _N() could be useful to tag internal error messages for which
//   translation is not required or desired
inline const char * string_internal_language(const char * CharPtr) { return CharPtr; }

#ifdef _N
#   error "cannot define alias _N() for function string_internal_language(), because _N is used as macro already"
#endif
// #define _N string_internal_language
inline const char * _N(const char * CharPtr) { return CharPtr; }

// string_no_localization()
// - tag strings as being some hard coded string constant
//   (not localizable at all)
inline const char * string_no_localization(const char * CharPtr) { return CharPtr; }

#ifdef _C
#   error "cannot define alias _C() for function string_no_localization(), because _C is used as macro already"
#endif
// #define _C string_internal_language
inline const char * _C(const char * CharPtr) { return CharPtr; }

//-------------------------------------------------------------------------

// WX_DEFINE_ARRAY_BYTE(unsigned char, wxByteArray);
// 
// bool loadFile (const wxString & FileNameString, 
//                wxByteArray & ByteArray);

//-------------------------------------------------------------------------

// NOTE semantics not well defined for Win16: there, some function like
// releaseDataResource() or copying the data block would be required
//
// This function is similar to wxLoadUserResource(), but it is intended
// to handle arbitrary data blocks; not just strings
//
// NOTE the 'ResourceTypeString' parameter must be a normal char pointer
// (not 'const wxString &'), because the RT_* definitions may not 
// be pointers to valid memory
extern bool loadDataResource (const wxString & ResourceNameString, 
                              const char * ResourceTypeString,
                              unsigned char ** DataPtr, size_t * DataSize);

//-------------------------------------------------------------------------

extern bool loadImageFromMemory (const unsigned char * ImageData, size_t ImageDataSize,
                                 wxImage & Image);
extern bool loadBitmapFromMemory (const unsigned char * ImageData, size_t ImageDataSize,
                                  wxBitmap ** Bitmap);

#if defined(__WXMSW__)

//-------------------------------------------------------------------------
// to put e.g. a PNG type image as it is (raw) into a resource, use e.g.:
//    IDR_About      RCDATA  DISCARDABLE     "images/About.png"
// and then just call:
//    wxBitmap AboutBitmap;
//    loadBitmapFromResource ("IDR_About", & AboutBitmap);
extern bool loadImageFromResource (const wxString & ImageDataResourceNameString, 
                                   wxImage & Image);
extern bool loadBitmapFromResource (const wxString & ImageDataResourceNameString, 
                                    wxBitmap ** Bitmap);

#endif // defined(__WXMSW__)

//-------------------------------------------------------------------------

#if (M_WxExtLib_IsUseGraphicsHelper == 1)
#  if wxUSE_GUI

bool resizeImageVGrid (const wxImage & InputImage,
                       wxImage & OutputImage,
                       int OutputWidth, int OutputHeight);

#  endif
#endif

//-------------------------------------------------------------------------

// doTextLineWrap
// - automatic line-break (wrap) using the font selected for the given DC
// - currently, line-break is possible only at character indices where
//   the input string contains a space (wxIsspace(char) used)
extern void doTextLineWrap (const wxString & InputString, 
                            int MaxLineWidth,
                            wxDC & DC,
                            wxString & ResultString);

extern void doTextLineWrap (const wxString & InputString, 
                            int MaxLineWidth,
                            wxWindow * Window,
                            wxString & ResultString);

//-------------------------------------------------------------------------

// wxVerifyWindowPtr()
// - search list/tree of windows recursively, starting from specified
//   ParentWindow
// - this realizes a depth-first search, no infinite recursions
//   should be possible with windows (tree-like structure),
//   unless windows are re-parented while running this function
bool wxVerifyWindowPtr (wxWindow * ParentWindow, wxWindow * SearchedWindow);

//-------------------------------------------------------------------------

// wxFocusRestorer
// - todo:
//   - optionally use wxVerifyWindowPtr() to make sure focus 
//     window is still there (needs to specify or find the top-most parent 
//     window first)
//   - also check that window is still enabled
class wxFocusRestorer
{
public:
    wxFocusRestorer ();
    ~wxFocusRestorer ();
private:
    wxWindow * m_OldFocusWindow;
};

//-------------------------------------------------------------------------

// wxMultiLineText:
// - use best suited control for output of possibly longer texts
//   (selected control depends on the used platform)
// - control implementations that can be used are wxTextCtrl and wxHtmlWindow
// - on platforms where wxTextCtrl doesn't support wrapping of long lines,
//   doTextLineWrap() is called automatically when calling setText()
// Rationale:
// - wxTextCtrl is under some platforms quite restricted:
//   - may not support wrapping of long lines (situation will change with 
//     wxWindows 2.6.0)
//   - may not support setting background and text colour
//   - under Lesstif: may not support setting a different font or font size
//   - may not support copying to clipboard (copying to clipboard works
//     for Motif/Lesstif, MSW, GTK)
// - wxHtmlWindow has everywhere the same functionality, but also
//   has some restrictions:
//   - copying to clipboard is not possible (situation will change with
//     wxWindows 2.6.0)
//   - probably lower performance than wxTextCtrl
//   (- might not have been compiled into the wxWindows library)
class wxMultiLineText
{
public:
    enum EMultiLineTextStyle
    {
      wxFORCE_TEXTCTRL    = 0x0001,
      wxFORCE_HTMLWINDOW  = 0x0002,

      wxHTML_MODE = 0x0010
    };

    wxMultiLineText ();
    // CtrlStyle can be: wxBORDER_NONE, wxSIMPLE_BORDER, wxSUNKEN_BORDER
    void create (wxWindow * ParentWindow,
                 int WindowIdent,
                 int CtrlStyle,
                 const wxSize & MinSize,
                 int MultiLineTextStyle = 0);

    // wxMultiLineText::getWindow() -> SetBackgroundColour()/SetForegroundColour()
    // may be used _before_ calling setText() is used - for wxHtmlWindow, suitable HTML 
    // code is generated to use those colors)
    // - if the wxHTML_MODE flag is specified, String must be in HTML
    //   (but without the <html>, <body> tags - they are added automatically)
    void setText (const wxString & String);
    wxString getHtmlTextString ();

    wxWindow * getWindow ();
    wxTextCtrl * getTextCtrl ();
    wxStaticText * getStaticText ();
    wxHtmlWindow * getHtmlWindow ();

    void setTextCtrl (wxTextCtrl * TextCtrl, int MultiLineTextStyle = 0);
    void getStaticText (wxStaticText * StaticText, int MultiLineTextStyle = 0);
    void setHtmlWindow (wxHtmlWindow * HtmlWindow, int MultiLineTextStyle = 0);

private:
    long m_MultiLineTextStyle;

    wxTextCtrl * m_TextCtrl;
    wxStaticText * m_StaticText; // never used - may not support scroll bar
    wxHtmlWindow * m_HtmlWindow;
    wxString m_HtmlTextString; // used only if type is wxHtmlWindow
};

//-------------------------------------------------------------------------

class wxTransparentStaticText : public wxStaticText
{
  public:
    enum ETransparentStaticTextFlags
      {
        // not really transparent :)
        IsOpaque = 0x0001,

        DefaultFlags = 0
      };

    wxTransparentStaticText();
    wxTransparentStaticText(wxWindow * ParentWindow,
                            wxWindowID ControlIdent,
                            const wxString& LabelString,
                            const wxPoint& Position = wxDefaultPosition,
                            const wxSize& Size = wxDefaultSize,
                            long Style = 0,
                            ETransparentStaticTextFlags Flags = DefaultFlags,
                            const wxString& Name = "TransparentStaticTextControl")
    {
        Create(ParentWindow, ControlIdent, LabelString, Position, Size, Style, Flags, Name);
    }

    bool Create(wxWindow * ParentWindow,
                wxWindowID ControlIdent,
                const wxString& LabelString,
                const wxPoint& Position = wxDefaultPosition,
                const wxSize& Size = wxDefaultSize,
                long Style = 0,
                ETransparentStaticTextFlags Flags = DefaultFlags,
                const wxString& Name = "TransparentStaticTextControl");

    void OnEraseBackground (wxEraseEvent & EraseEvent);
    void OnPaint (wxPaintEvent & PaintEvent);

  protected:
    ETransparentStaticTextFlags m_TransparentStaticTextFlags;

  private:
    DECLARE_DYNAMIC_CLASS (wxTransparentStaticText)
    DECLARE_EVENT_TABLE()
};

//-------------------------------------------------------------------------

// wxColourBarWindow:
// - a control/rectangle filled with the specified colour
// - the base class used has been switched from wxWindow to wxPanel,
//   because if a sizer and controls are placed "onto" the wxColourBarWindow,
//   sizing of the sizer doesn't seem to work correctly if base class is
//   just wxWindow
class wxColourBarWindow : public wxPanel
{
  public:
    wxColourBarWindow ();
    wxColourBarWindow (wxWindow * ParentWindow, int ControlIdent,
                       const wxPoint & Position = wxDefaultPosition, 
                       const wxSize & Size = wxDefaultSize,
                       long Style = wxTAB_TRAVERSAL | wxNO_BORDER, const wxString& Name = wxPanelNameStr);
    wxColourBarWindow (wxWindow * ParentWindow, int ControlIdent,
                       const wxColour & Colour,
                       const wxPoint & Position = wxDefaultPosition, 
                       const wxSize & Size = wxDefaultSize,
                       long Style = wxTAB_TRAVERSAL | wxNO_BORDER, const wxString& Name = wxPanelNameStr);

    void setBrush (const wxBrush & Brush);
    void setPen (const wxPen & Pen);
    void setRoundedCornerRadius (double RoundedCornerRadius = 2.);

    void OnEraseBackground (wxEraseEvent & EraseEvent);
    void OnPaint (wxPaintEvent & PaintEvent);

  private:
    wxBrush m_Brush;
    wxPen m_Pen;
    double m_RoundedCornerRadius;

    DECLARE_DYNAMIC_CLASS (wxColourBarWindow)
    DECLARE_EVENT_TABLE()
};

//-------------------------------------------------------------------------

#if (M_WxExtLib_IsUseGraphicsHelper == 1)

// wxScaledImageWindow:
// - a control/rectangle filled with the specified colour
// - the base class used is wxPanel (see note in wxColourBarWindow)
// - requires resizeImageVGrid() from GraphicsHelper (GraphicsMisc.h)
class wxScaledImageWindow : public wxPanel
{
  public:
    wxScaledImageWindow ();
    wxScaledImageWindow (wxWindow * ParentWindow, int ControlIdent,
                         const wxPoint & Position = wxDefaultPosition, 
                         const wxSize & Size = wxDefaultSize,
                         long Style = wxTAB_TRAVERSAL | wxNO_BORDER, const wxString& Name = wxPanelNameStr);
    wxScaledImageWindow (wxWindow * ParentWindow, int ControlIdent,
                         const wxImage & Image,
                         const wxPoint & Position = wxDefaultPosition, 
                         const wxSize & Size = wxDefaultSize,
                         long Style = wxTAB_TRAVERSAL | wxNO_BORDER, const wxString& Name = wxPanelNameStr);

    void setImage (const wxImage & Image);

    void OnEraseBackground (wxEraseEvent & EraseEvent);
    void OnPaint (wxPaintEvent & PaintEvent);

 private:
    wxImage m_OriginalImage;

    bool m_IsCacheOkay;
    wxRect m_PreviousClientRect;
    wxBitmap m_ScaledBitmap;

    DECLARE_DYNAMIC_CLASS (wxScaledImageWindow)
    DECLARE_EVENT_TABLE()
};

#endif

//-------------------------------------------------------------------------

class wxStatusMessageTarget;

// wxItemWindow:
// A control which can be selected and focused. The selection and focus
// states are visualized by drawing colored rectangles around the
// actual item drawing area/rectangle. This kind of selection and focus
// visualization may be useful for e.g. images (which may not look nice
// when inverted, blue-ified etc.).
// 
// Note that the focus state is (intentionally) independent of the 
// the wxWindow focus.
// 
// Currently this class doesn't handle selection and focus state changes 
// from input events itself. However, a derived class may define e.g. 
// an OnMouseEvent() handler to handle these state changes.
//
class wxItemWindow : public wxWindow
{
  public:
    wxItemWindow ();
    wxItemWindow (wxWindow * ParentWindow, int ControlIdent,
                      const wxPoint & Position = wxDefaultPosition, 
                      const wxSize & Size = wxDefaultSize,
                      bool IsSelectable = false, bool IsFocusable = false,
                      wxStatusMessageTarget * StatusMessageTarget = NULL,
                      long Style = 0, const wxString& Name = wxPanelNameStr);

    void init ();

    void setFocusedBackgroundBrush (const wxBrush & Brush);

    void OnEraseBackground (wxEraseEvent & EraseEvent);
    void OnPaint (wxPaintEvent & PaintEvent);

    virtual void handleItemPaint (wxPaintDC & PaintDC,
                                  const wxRect & ItemRect,
                                  const wxBrush * BackgroundBrush = NULL);

    wxStatusMessageTarget * m_StatusMessageTarget;
    bool m_IsSelectable;
    bool m_IsSelected;
    bool m_IsFocusable;
    bool m_IsFocused;
    bool m_IsBorder;

    bool m_IsUsingFocusedBackgroundBrush;
    wxBrush m_FocusedBackgroundBrush;

  private:
    DECLARE_DYNAMIC_CLASS (wxItemWindow)
    DECLARE_EVENT_TABLE()
};

//-------------------------------------------------------------------------

// NOTE EResolutionIdent is not yet used
enum EResolutionIdent
{
    FullResolution,
    ThumbnailResolution
};

#if defined(__WXMSW__)

// currently MSW only
class wxImageProxy
{
  public:
    wxImageProxy ();
    ~wxImageProxy ();

    virtual void getDIB (EResolutionIdent ResolutionIdent, WXHDIB & DIB) = 0;
    virtual void draw (wxDC & DC, int X, int Y, int Width, int Height,
                       EResolutionIdent ResolutionIdent,
                       const wxBrush * BackgroundBrush = NULL);

    virtual void updateFromFile (bool IsReadingForced = false) = 0;
    virtual void discard ();

    //-------------------------------------------------------------------------
    virtual void loadFromFile (const wxString & PathString) = 0;
    virtual void freeDIB ();

    WXHDIB m_DIBFromFile;
};

#endif // defined(__WXMSW__)


//-------------------------------------------------------------------------

#if defined(__WXMSW__)

// wxBitmapPaintHelper:
// - facilitate dealing with memory DCs under MSW (create a compatible DC,
//   automatic deselection of image from DC)
//
// currently MSW only:
class wxBitmapPaintHelper
{
public:
    wxBitmapPaintHelper (const wxString & ResourceIdentString);
    wxBitmapPaintHelper (const wxString & ResourceIdentString, HDC PaintDCHandle);
    wxBitmapPaintHelper (wxBitmap & Bitmap);
    wxBitmapPaintHelper (wxBitmap & Bitmap, HDC PaintDCHandle);
    wxBitmapPaintHelper (int Width, int Height);
    wxBitmapPaintHelper (int Width, int Height, HDC PaintDCHandle);
    ~wxBitmapPaintHelper ();

    void createBitmap (int Width, int Height);
    void setBitmap (wxBitmap & Bitmap);
    void loadBitmap (const wxString & ResourceIdent);
    void prepare (HDC PaintDCHandle);
    void release ();

    int getWidth ();
    int getHeight ();

    wxBitmap m_Bitmap;
    wxBitmap * m_BitmapRef;
    HDC m_MemoryDCHandle;
    HBITMAP m_BitmapHandle;
    HBITMAP m_PreviousBitmapHandle;
};

#endif // defined(__WXMSW__)

//-------------------------------------------------------------------------

#if defined(__WXMSW__)

// DIB only
extern int getActualUsedColorCount (BITMAPINFO * BitmapInfo);
extern void drawDIBFitted (wxDC & WxDC, int X, int Y, int Width, int Height,
                           WXHDIB DIBHandle,
                           const wxBrush * BackgroundBrush = NULL);

#endif // defined(__WXMSW__)

//-------------------------------------------------------------------------

// WxBeep
// - emit a beep for about the given number of milli-seconds
extern void WxBeep (int MilliSeconds);

//-------------------------------------------------------------------------

// doRotateMirrorCoord() - rotates and mirrors points (x,y) around the
// rotation center whose coordinates determine the mirroring axes
// Notes:
// 1) RotationAngle must be either 0, 90, 180 or 270 - otherwise the 
//    behavior is undefined
// 2) Vertical mirroring axis is a horizontal line at CentrePoint.y,
//    horizontal mirroring axis is a vertical line at CentrePoint.y
// 3) Order of application: mirroring is applied after rotation, 
//    not before
extern void doRotateMirrorCoord (int RotationAngle, 
                                 bool IsHorizontallyMirrored, 
                                 bool IsVerticallyMirrored,
                                 const wxPoint & CentrePoint,
                                 const wxPoint & InputPoint,
                                 wxPoint & OutputPoint,
                                 bool IsRotateCenter = false);

//-------------------------------------------------------------------------

class wxRotationMirrorState
{
public:
    wxRotationMirrorState();
    ~wxRotationMirrorState();

    void reset ();

    // maybe just one mirroring axis would be sufficient
    void mirror (bool IsHorizontallyMirrored, bool IsVerticallyMirrored);
    void rotate (int RotationAngle);
    void handleMirrorRotation(int RotationAngle,
                              bool & IsHorizontallyMirrored,
                              bool & IsVerticallyMirrored);

    void getNormalizedState (int & RotationAngle, 
                             bool & IsHorizontallyMirrored, 
                             bool & IsVerticallyMirrored);

    void assign (const wxRotationMirrorState & RHSRotationMirrorState);
    void subtract (const wxRotationMirrorState & FirstRotationMirrorState,
                   const wxRotationMirrorState & SecondRotationMirrorState);

    bool m_IsUpdateRequired;

    bool m_IsHorizontallyMirrored;
    bool m_IsVerticallyMirrored;
    int m_RotationAngle;
};

//-------------------------------------------------------------------------

#ifdef M_WxExtLib_IsFilteredDIBEnabled

// try to forward-declare ImageMagick's image and ExceptionInfo:
extern "C"
{
  struct _Image; // wxImageMagick image
  struct _ExceptionInfo; // wxImageMagick ExceptionInfo
}

// WX_DECLARE_EXPORTED_BASEARRAY(char, wxBaseArrayChar);
// _WX_DEFINE_BASEARRAY (char, wxBaseArrayChar);

// WX_DECLARE_BASEARRAY(unsigned char, wxBaseArrayUChar);
// _WX_DEFINE_BASEARRAY (unsigned char, wxBaseArrayUChar);

// WARN how will ExportMagic (redefined class, extern keywords)
// interact with this wx-macro?
//
WX_DEFINE_ARRAY_LONG(long, wxFilteredDIBColorPaletteArray);

#if defined(__WXMSW__)

// currently DIB only:

// wxFilteredDIB:
// - input and output image format is DIB
// - filtering may be realized by applying an ImageMagick filter
// - rotation is done in DIB format and must be handled by a derived class
// - wxFilteredDIB will manage required conversions (e.g. convert from a 
//   palette-based DIB to 32 Bit color ImageMagick if any filter is to 
//   be applied)
// - for palette-based images, color adjustment is realized by modifying the
//   palette in-place which is much faster than modifying all pixels - the
//   old palette is stored for a later undo or a new modification
// - image rotation must be 0, 90, 180 or 270 degree - if only the rotation
//   angle (or mirroring) has changed since last filtering, 
//   rotation/mirroring will be made relative to the last result 
//   (improves performance)
// Notes:
// - ImageMagick's image type also supports palette-based images, but
//   many filters will only work for 32 Bit images (or 64 Bit when using
//   16-Bit color channel resolution)
// - linking of the ImageMagick core library (the C-API) is required
//   to use wxFilteredDIB
// - the files 'ImageMagickExtensions.{h,c}' are required for DIB <-> 
//   ImageMagick image and color depth conversions
// Todo:
// - implement better color adjustment function
// - implement DIB rotation functions (or look for existing code)
// - check new wxDIB class (new since wxWidgets 2.5.?)
class wxFilteredDIB
{
public:
    wxFilteredDIB ();
    ~wxFilteredDIB ();

    // color adjustment
    // Brightness [-1.0, 1.0], Contrast [-1.0, 1.0]
    void setBrightnessContrast (double Brightness, double Contrast);
    void setColorAdjustment (double RedGamma, double GreenGamma,
                             double BlueGamma);
    void setInversion (bool IsInverted);

    // palette based color adjustment
    // virtual void setPaletteBasedColorAdjustmentEnabled (bool IsEnable);
    virtual void rememberOriginalPalette ();
    virtual void rememberModifiedPalette ();
    virtual void setOriginalPaletteIntoSourceImage ();
    virtual void setModifiedPaletteIntoSourceImage ();
    bool rememberPalette (WXHDIB DIBHandle,
                          wxFilteredDIBColorPaletteArray & ColorPaletteByteArray);
    bool restorePalette (WXHDIB DIBHandle,
                         wxFilteredDIBColorPaletteArray & ColorPaletteByteArray);
    virtual bool getIsUsePaletteAlienation();
    virtual bool applyPaletteAlienation (wxFilteredDIBColorPaletteArray & ColorPaletteByteArray);

    // filtering related
    virtual void setIsFilteringModified ();
    virtual void setIsImageMagickFilteringRequired (bool IsRequired);

    // rotation and mirroring
    virtual void resetRotationMirror ();
    virtual void rotate (int RotationAngle);
    virtual void mirror (bool IsHorizontallyMirrored, bool IsVerticallyMirrored);

    // setting source image, obtaining result image
    virtual void setSourceImage (const WXHDIB & SourceDIBHandle);
    virtual void getResultImage (WXHDIB & ResultDIBHandle);
    virtual void setIsOwningSourceImage (bool IsOwningSourceImage);
    virtual void setIsOwningResultImage (bool IsOwningResultImage);
    void freeSourceImage();
    void freeResultImage();

    // updating result image
    virtual void updateResultImage ();
    virtual void handleResultImageChanged ();

    // handler which do actual image modifications/transformations
    virtual void applyColorAdjustment (struct _Image ** MagickImagePtrPtr,
                                       struct _ExceptionInfo * exceptionInfo);
    virtual void applyImageMagickFiltering (struct _Image ** MagickImagePtrPtr,
                                            struct _ExceptionInfo * exceptionInfo);
    virtual void applyImageRotationMirroring (int RotationAngle,
                                              bool IsHorizontallyMirrored,
                                              bool IsVerticallyMirrored,
                                              WXHDIB & InputDIBHandle,
                                              WXHDIB & OutputDIBHandle,
                                              bool & IsNewOutputDIBHandle);

    virtual WXHDIB applyDIBRotation (WXHDIB InputDIBHandle, int RotationAngle) = 0;
    virtual WXHDIB applyDIBMirroring (WXHDIB InputDIBHandle, bool IsVertically) = 0;

protected:
    // color adjustment
    double m_Brightness;
    double m_Contrast;
    double m_RedGamma;
    double m_GreenGamma;
    double m_BlueGamma;
    bool m_IsInverted;
    bool m_IsColorAdjustmentModified;
    void levelGamma (double & Gamma);

    // palette based color adjustment
    bool m_IsPaletteBasedColorAdjustmentEnabled;
    wxFilteredDIBColorPaletteArray m_OriginalColorPaletteByteArray;
    bool m_IsOriginalColorPaletteRemembered;
    wxFilteredDIBColorPaletteArray m_ModifiedColorPaletteByteArray;
    bool m_IsModifiedColorPaletteRemembered;

    // filtering related
    bool m_IsImageMagickFilteringRequired;
    bool m_IsFilteringModified;
    int m_FilteredDIBBitsPerPixel;

    // rotation and mirroring
    wxRotationMirrorState m_PreviousRotationMirrorState;
    wxRotationMirrorState m_RotationMirrorState;

    // source image, result image handling
    void freeImage (WXHDIB & DIBHandle);
    WXHDIB m_SourceDIBHandle;
    bool m_IsOwningSourceImage;
    WXHDIB m_ResultDIBHandle;
    bool m_IsOwningResultImage;
};

#endif // defined(__WXMSW__)

#endif

//-------------------------------------------------------------------------

// wxChoiceManager:
// - manages selections from a set of source items into
//   choice items
class wxChoiceManager
{
public:
    wxChoiceManager ();

    // high-level user interface interaction
    void OnSelect ();
    void OnDeselect ();
    void OnClearSelection ();
    void OnSourceLeftMouseButton (int SourceItemIndex);
    void OnSourceRightMouseButton (int SourceItemIndex);
    void OnChoiceLeftMouseButton (int ChoiceItemIndex);
    void OnChoiceRightMouseButton (int ChoiceItemIndex);

    //-------------------------------------------------------------------------
    // selection and focus managment
    // almost all functions accept '-1' as notion of 'no document' or
    // 'no item'

    void handleSelection (bool IsNewlyFocused, int ChoiceItemIndex,
                          int SourceItemIndex);

    void selectIntoChoice (int ChoiceItemIndex, int SourceItemIndex,
                            bool IsUpdateChoiceView = true);
    void deselectFromChoice (int ChoiceItemIndex, bool IsUpdateChoiceView = true);

    // focus management
    int getSourceFocus();
    void setSourceFocus (int SourceItemIndex);
    virtual void updateSourceFocus (int NewFocusedItemIndex, bool IsFocus) = 0;
    int getChoiceFocus();
    void setChoiceFocus (int ChoiceItemIndex);
    virtual void updateChoiceFocus (int NewFocusedItemIndex, bool IsFocus) = 0;

    // this should be kept in sync with m_IsFocused of e.g. wxItemWindow
    int m_FocusedChoiceItemIndex;
    int m_FocusedSourceItemIndex;

    // selection managment
    void redoSelections ();
    void setSelection (int SourceItemIndex, bool IsSelect);
    virtual void updateSelection (int SourceItemIndex, bool IsSelect) = 0;

    virtual void updateChoiceView () = 0;

    // functions for mapping from item index into document ident or vice
    // versa
    virtual int getSourceDocument (int SourceItemIndex) = 0;
    virtual int getChoiceDocument (int ChoiceItemIndex) = 0;
    virtual int getSourceItemCount () = 0;
    virtual int getChoiceItemCount () = 0;

    virtual int findSourceItemIndex (int DocumentIdent);
    virtual int findChoiceItemIndex (int DocumentIdent);

    virtual void insertChoiceDocument (int DocumentIdent, int ChoiceItemIndex, int Count) = 0;
    virtual void setChoiceDocument (int ChoiceItemIndex, int DocumentIdent) = 0;
};

//-------------------------------------------------------------------------

typedef wxWindow * (* wxDefaultParentWindowFunc) (wxWindow *);

// set or get global pointer to wxDefaultParentWindowFunc:
void wxSetGlobalDefaultParentWindowFunc (wxDefaultParentWindowFunc Func);
wxDefaultParentWindowFunc wxGetGlobalDefaultParentWindowFunc ();

// wxGetDefaultParentWindow():
// - just return ParentWindow if ParentWindow != NULL, or determine
//   a default parent window by calling wxGetGlobalDefaultParentWindowFunc
//   (if set)
// - note: this function may still return NULL
wxWindow * wxGetDefaultParentWindow (wxWindow * ParentWindow);

//-------------------------------------------------------------------------

#define M_IsUndefExportMagic 1
#   include "WxExtLibAliases.h"
#undef M_IsUndefExportMagic

//-------------------------------------------------------------------------

#endif
