
/*
-------------------------------------------------------------------------
This file is part of WxWidgetsExtensions library.
-------------------------------------------------------------------------

WxExtLib (WxWidgetsExtensions) library
-----------------------------

COPYRIGHT NOTICE:

WxExtLib library Copyright (c) 2003-2007 Daniel Kps

The WxWidgetsExtensions library and associated documentation files (the
"Software") is provided "AS IS".  The author(s) disclaim all
warranties, expressed or implied, including, without limitation, the
warranties of merchantability and of fitness for any purpose.  The
author(s) assume no liability for direct, indirect, incidental,
special, exemplary, or consequential damages, which may result from
the use of or other dealings in the Software, even if advised of the
possibility of such damage.

Permission is hereby granted, free of charge, to any person obtaining
a copy of this Software, to deal in the Software without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Software,
and to permit persons to whom the Software is furnished to do so,
subject to the following conditions:

 1. The origin of this source code must not be misrepresented.
 2. Altered versions must be plainly marked as such and must not be
    misrepresented as being the original source.
 3. This Copyright notice may not be removed or altered from any 
    source or altered source distribution.

End of WxExtLib library Copyright Notice

-------------------------------------------------------------------------
*/

#ifndef _NO_HEADER_INCLUDE

#if defined(__GNUG__) && (!defined(__APPLE__)) && (!(defined M_NoPragmaInterface))
    #pragma implementation "WxMisc.cpp"
#endif

#include "WxExtLibConfig.h"

// For compilers that support precompilation, includes "wx/wx.h".
#include "wx/wxprec.h"

#ifdef __BORLANDC__
    #pragma hdrstop
#endif

#include "WxMisc.h"

#include <wx/mstream.h>

#if wxUSE_GUI
#   include <wx/dc.h>
#   include <wx/dcclient.h>

#   ifdef M_WxExtLib_IsFilteredDIBEnabled
#       include "magick/studio.h"
#       include "magick/error.h"
#       include "magick/image.h"
#       include "magick/enhance.h"
#       include "magick/constitute.h"
#       include "ImageMagickExtensions.h"
#   endif
#endif // wxUSE_GUI

#include "safecast.h"

#if (M_WxExtLib_IsUseGraphicsHelper == 1)
#   include <GraphicsMisc.h>
#endif

#endif // _NO_HEADER_INCLUDE

//=========================================================================

wxMinimalTimeInterval::wxMinimalTimeInterval ()
{
    m_IsInitialized = false;
}
    
void wxMinimalTimeInterval::setMinimalTimeInterval (long MinimalTimeIntervalMS)
{
    m_MinimalTimeIntervalMS = MinimalTimeIntervalMS;
}

void wxMinimalTimeInterval::setBegin ()
{
#ifdef M_MinimalTimeInterval_UseGetTickCount
    m_BeginUpTimeMS = ::GetTickCount();
#else
    // m_StopWatch.Start ();
    m_BeginTimeMSLongLong = wxGetLocalTimeMillis();
#endif
    m_IsInitialized = true;
}

bool wxMinimalTimeInterval::check ()
{
    // return true if start of interval is unspecified, i.e. no start event
    // has taken place
    if (!m_IsInitialized)
      return true;

#ifdef M_MinimalTimeInterval_UseGetTickCount
    DWORD CurrentUpTimeMS = ::GetTickCount ();
    DWORD ElapsedTimeMS;
    if (m_BeginUpTimeMS > CurrentUpTimeMS)
      {
        // handle timer overflow?
//      CurrentUpTimeMS += 0xFFFFFFFF - CurrentUpTimeMS;
//      m_BeginUpTimeMS = 0;
        ElapsedTimeMS = CurrentUpTimeMS - m_BeginUpTimeMS;
      }
    else
      {
        ElapsedTimeMS = CurrentUpTimeMS - m_BeginUpTimeMS;
      }
    
    // if (ElapsedTimeMS >= m_MinimalTimeIntervalMS)
    if (cast_is_equal_smaller (m_MinimalTimeIntervalMS, ElapsedTimeMS))
      {
        return true;
      }
    else
      {
        return false;
      }
#else
    // BUG doesn't handle timer overflows (will happen after 
    // about 48 days)
    // return (m_StopWatch.Time () > m_MinimalTimeIntervalMS);

    // alternative implementation: use wxGetLocalTimeMillis() which
    // uses wxLongLong and thus has no overflow for about eternal years
    wxLongLong CurrentTimeMSLongLong = wxGetLocalTimeMillis();
    wxLongLong MinimalTimeIntervalMSLongLong = m_MinimalTimeIntervalMS;
    wxLongLong ElapsedTimeMSLongLong = CurrentTimeMSLongLong - m_BeginTimeMSLongLong;

    // WARN don't know if ElapsedTimeMS<0 may happen - under e.g. Win9x this seems to be 
    // the case after adjusting the system clock backwards; however, adjusting
    // the system clock forwards will yield incorrect results too -
    // TODO check if adjusting system time influences running processes under 
    // Linux, Unix
    // WARN workaround for ElapsedTimeMS < 0
    if (ElapsedTimeMSLongLong < 0)
      {
        setBegin();
        return false;
      }
    if (ElapsedTimeMSLongLong >= MinimalTimeIntervalMSLongLong)
      {
        return true;
      }
    else
      {
        return false;
      }
#endif
}

//=========================================================================

bool convertStringToLong (const char * CharPtr, long & Long)
{
    // replacement for:
    // if (sscanf (CharPtr,"%d%c", & TempLong, &DummyChar) !=1 ) { // handle error }

    bool IsOk = false;
    char * EndCharPtr = NULL;
	int Base = 10;
    long TempLong = strtol (CharPtr, & EndCharPtr, Base);

    if (EndCharPtr != NULL 
		&& EndCharPtr != CharPtr 
		&& *EndCharPtr == '\0')
      {
        // WARN should use macros for minimal/maximal long values
        // defined in some header file somewhere
		// using macros LONG_MAX and LONG_MIN may require #include <limits.h>
        if ((TempLong != LONG_MAX)
			&& (TempLong != LONG_MIN))
		  {
			IsOk = true;
			Long = TempLong;
		  }
	  }
	return IsOk;
}

// same as above, but for 'unsigned long' instead of 'long'
bool convertStringToLong (const char * CharPtr, unsigned long & Long)
{
    // replacement for:
    // if (sscanf (CharPtr,"%u%c", & TempLong, &DummyChar) !=1 ) { // handle error }

    bool IsOk = false;
    char * EndCharPtr = NULL;
	int Base = 10;
    unsigned long TempLong = strtoul (CharPtr, & EndCharPtr, Base);

    if (EndCharPtr != NULL 
		&& EndCharPtr != CharPtr 
		&& *EndCharPtr == '\0')
      {
		// using macros ULONG_MAX may require #include <limits.h>
        if ((TempLong != ULONG_MAX)
			&& (TempLong != 0))
		  {
			IsOk = true;
			Long = TempLong;
		  }
	  }
	return IsOk;
}

//=========================================================================

void convertAsciiToHtml (const char * AsciiStringCharPtr, wxString & HtmlString,
                         bool IsTranslateNewlines,
                         long WxAsciiToHtmlFlags)
{
    int HtmlStringLength = 0;
    HtmlString.Clear();

#   define M_HandleReplace(HtmlCharSequence) \
    \
    if (StageIndex == 0) \
      { \
        /* PERFORMANCE calling strlen() should be replaced by some */ \
        /* static string length table */ \
        HtmlStringLength += strlen(HtmlCharSequence); \
      } \
    else                                       \
      { \
        HtmlString += HtmlCharSequence; \
      }

#   define M_HandleNonReplace \
    \
    if (StageIndex == 0) \
      { \
        HtmlStringLength += 1; \
      } \
    else                                       \
      { \
        HtmlString += (char) CharAsInt; \
      }

    // now the loop in two stages:
    // - first stage:
    //   determine length of resulting HTML string and pre-alloc to
    //   this length to avoid multiple reallocations
    // - second stage:
    //   do actual replacement/append
    for (int StageIndex = 0; StageIndex < 2; ++StageIndex)
      {
        bool IsLeadingSpace = true;

        if (StageIndex == 1)
          {
            HtmlString.Alloc (HtmlStringLength);
          }
    
        const char * CurrentAsciiStringCharPtr = NULL;

        // loop over all characters in input string
        for (CurrentAsciiStringCharPtr = AsciiStringCharPtr; 
             *CurrentAsciiStringCharPtr!='\0'; ++CurrentAsciiStringCharPtr)
          {
            char Char = *CurrentAsciiStringCharPtr;
            int CharAsInt = Char;

            bool IsSpaceChar = false;

            switch (CharAsInt)
              {
              case '<':
                M_HandleReplace ("&lt;");
                break;
              case '>':
                M_HandleReplace ("&gt;");
                break;
              case '&':
                M_HandleReplace ("&amp;");
                break;
              case '"':
                M_HandleReplace ("&quot;");
                break;
              case ' ': 
              case '\t': 
                IsSpaceChar = true;
                if ((IsLeadingSpace && ((WxAsciiToHtmlFlags & wxAsciiToHtml_ConvertLeadingSpaces) != 0))
                     || ((WxAsciiToHtmlFlags & wxAsciiToHtml_ConvertAllSpaces) != 0))
                  {
                    if (Char == '\t')
                      {
                        M_HandleReplace ("&nbsp;&nbsp;&nbsp;&nbsp;");
                      }
                    else
                      {
                        M_HandleReplace ("&nbsp;");
                      }
                  }
                else
                  {
                    M_HandleNonReplace;
                  }
                break;
              case '\r':
                if (IsTranslateNewlines)
                  {
                    // it is assumed that next character it '\n' and this 
                    // will cause "<br>\r\n" to be appended
                    M_HandleReplace ("");
                  }
                else
                  {
                    M_HandleNonReplace;
                  }
                break;
              case '\n':
                if (IsTranslateNewlines)
                  {
                    M_HandleReplace ("<br>\r\n");

                    IsSpaceChar = true;
                    IsLeadingSpace = true;
                  }
                else
                  {
                    M_HandleNonReplace;
                  }
              default:
                M_HandleNonReplace;
                break;
              }

            if (IsLeadingSpace && (!IsSpaceChar))
              {
                IsLeadingSpace = false;
              }
          }
      }

#undef M_HandleReplace
#undef M_HandleNonReplace

}

wxString convertAsciiToHtml (const char * AsciiStringCharPtr,
                             bool IsTranslateNewlines,
                             long WxAsciiToHtmlFlags)
{
    wxString HtmlString;
    convertAsciiToHtml (AsciiStringCharPtr, HtmlString, 
                        IsTranslateNewlines,
                        WxAsciiToHtmlFlags);

    return HtmlString;
}

//-------------------------------------------------------------------------

wxString formatColourToHtmlParam (const wxColour & Colour)
{
    return wxString::Format ("#%02x%02x%02x", 
                             Colour.Red(), Colour.Green(), Colour.Blue());
}

//=========================================================================

bool getHtmlTagParamAsDouble (const wxHtmlTag & HtmlTag,
                              const wxString & ParamNameString, double * Double)
{
    wxString ParamValueString = HtmlTag.GetParam (ParamNameString);
    return parseDoubleFromAnyLocaleString (ParamValueString, Double);
}

bool parseDoubleFromAnyLocaleString (const wxString & InputValueString, double * Double)
{
    // HACK to get around problem with locale for float/double formatting
    // wxString::Format() and HtmlTag::ScanParam() seem to use current locale 
    // settings, but sometimes we want to parse (as strings) hard-coded double 
    // parameters (e.g. occuring in HTML templates)
    // Normally ScanParam() would fail (depending on the locale setting).

    // BUG
    // 1) the used approach probably fails if the input strings contains 
    //    thousands-separators (grouping characters)
    // 2) the used approach probably fails if a comma is used to separate
    //    a sequence of numbers (possibly written without spaces)

    // WARN
    // - don't know if there are other decimal separator charcaters than
    //   comma and point. If so, parsing for them would have to be added too.

    wxString ValueString (InputValueString);

    // try to parse in two different ways: once with occurences of
    // points replaced by commas and once with occurences of commas
    // replaced by points
    int WithCommaParsedCharacterCount = 0;
    double WithCommaParsedDouble = 0.;
    ValueString.Replace (".", ",", true);
    int WithCommaParsedArgumentCount = wxSscanf (ValueString.c_str(), "%lf%n", 
                                                 & WithCommaParsedDouble,
                                                 & WithCommaParsedCharacterCount);

    int WithPointParsedCharacterCount = 0;
    double WithPointParsedDouble = 0.;
    ValueString.Replace (",", ".", true);
    int WithPointParsedArgumentCount = wxSscanf (ValueString.c_str(), "%lf%n", 
                                                 & WithPointParsedDouble,
                                                 & WithPointParsedCharacterCount);

    // determine which of both possible results to use: if both ways we could
    // parse at least on parameter, we use the result where more characters were
    // parsed (quite hacky, admittedly). It would be better to ask for 
    // the setting of the locale, but i don't know which function is available
    // (if any) for this.
    bool IsUsePointParsedDouble = true;
    if (WithPointParsedArgumentCount > 0
        && WithPointParsedArgumentCount == 0)
      {
        IsUsePointParsedDouble = true;
      }
    else if (WithPointParsedArgumentCount == 0
        && WithCommaParsedArgumentCount > 0)
      {
        IsUsePointParsedDouble = false;
      }
    else if (WithPointParsedArgumentCount > 0
        && WithCommaParsedArgumentCount > 0)
      {
        IsUsePointParsedDouble = (WithPointParsedCharacterCount >= WithCommaParsedCharacterCount);
      }

    // set function's output values
    int ParsedArgumentCount = IsUsePointParsedDouble 
      ? WithPointParsedArgumentCount : WithCommaParsedArgumentCount;
    if (IsUsePointParsedDouble)
      {
        * Double = WithPointParsedDouble;
      }
    else
      {
        * Double = WithCommaParsedDouble;
      }

    return ParsedArgumentCount > 0;
}

//=========================================================================

wxNumberFormatOptions::wxNumberFormatOptions ()
{
    m_GroupSize = 3;
    m_GroupSeparatorChar = ',';
    m_DecimalSeparatorChar = '.';
}

wxNumberFormatOptions::wxNumberFormatOptions (int GroupSize, char GroupSeparatorChar, 
                                              char DecimalSeparatorChar)
{
    set (GroupSize, GroupSeparatorChar, DecimalSeparatorChar);
}

void wxNumberFormatOptions::set (int GroupSize, char GroupSeparatorChar, 
                                 char DecimalSeparatorChar)
{
    m_GroupSize = GroupSize;
    m_GroupSeparatorChar = GroupSeparatorChar;
    m_DecimalSeparatorChar = DecimalSeparatorChar;
}

//-------------------------------------------------------------------------

wxNumberFormattingInternals::wxNumberFormattingInternals()
{
    m_IsOkay = false;
}

//-------------------------------------------------------------------------

void detectNumberFormattingInternals(wxNumberFormattingInternals * NumberFormattingInternals)
{
    wxString TempString;

    bool IsOkay = false;

    for_once
      {
        // determine decimal separator string
        double SampleDouble = 1.4;

        TempString = wxString::Format ("%f", SampleDouble);
        const char * TempCharPtr = TempString.c_str();

        if (TempCharPtr[1] == ',')
          NumberFormattingInternals -> m_DecimalSeparatorChar = ',';
        else if (TempCharPtr[1] == '.')
          NumberFormattingInternals -> m_DecimalSeparatorChar = '.';
        else 
          break;

        if ((TempCharPtr[0] != '1') || (TempCharPtr[2] != '4'))
          break;

        IsOkay = true;
      }

    NumberFormattingInternals -> m_IsOkay = IsOkay;
}

//-------------------------------------------------------------------------

void formatLong (const wxString & FormatString, 
                 wxNumberFormatOptions & NumberFormatOptions, 
                 long Long, 
                 wxString & String, 
                 wxNumberFormattingInternals * NumberFormattingInternals)
{
    // example: -123456.789 -> -123,456.789

    if (NumberFormattingInternals == NULL)
      {
        wxNumberFormattingInternals TempNumberFormattingInternals;
        detectNumberFormattingInternals (& TempNumberFormattingInternals);
        NumberFormattingInternals = & TempNumberFormattingInternals;
      }

    if (!NumberFormattingInternals -> m_IsOkay)
      {
        String = wxString::Format (FormatString, Long);
        return;
      }

    // format into temporary string and add grouping if desired
    wxString TempString;
    TempString = wxString::Format (FormatString, Long);

    formatNumberStringWithGrouping (TempString,
                                    NumberFormatOptions,
                                    String,
                                    NumberFormattingInternals);
}

void formatDouble (const wxString & FormatString, 
                   wxNumberFormatOptions & NumberFormatOptions, 
                   double Double, 
                   wxString & String, 
                   wxNumberFormattingInternals * NumberFormattingInternals)
{
    // example: -123456.789 -> -123,456.789

    wxNumberFormattingInternals TempNumberFormattingInternals;
    if (NumberFormattingInternals == NULL)
      {
        detectNumberFormattingInternals (& TempNumberFormattingInternals);
        NumberFormattingInternals = & TempNumberFormattingInternals;
      }

    if (!NumberFormattingInternals -> m_IsOkay)
      {
        String = wxString::Format (FormatString, Double);
        return;
      }

    // format into temporary string and add grouping if desired
    wxString TempString;
    TempString = wxString::Format (FormatString, Double);

    formatNumberStringWithGrouping (TempString,
                                    NumberFormatOptions,
                                    String,
                                    NumberFormattingInternals);
}

//-------------------------------------------------------------------------

void formatNumberStringWithGrouping (const wxString & NumberString, 
                                     wxNumberFormatOptions & NumberFormatOptions, 
                                     wxString & String, 
                                     wxNumberFormattingInternals * NumberFormattingInternals)
{
    /* from manual page of fprintf():
       For some numeric conversions a radix character (`decimal point') or thousands' grouping character is used. The
       actual character used depends on the LC_NUMERIC part of the locale. The POSIX locale uses `.' as radix charac-
       ter, and does not have a grouping character.  Thus,
                   printf("%'.2f", 1234567.89);
       results in `1234567.89' in the POSIX locale, in `1234567,89' in the nl_NL locale, and in `1.234.567,89' in the
       da_DK locale.

       thus:
       - newer version of the standard library may have support for using thousand grouping
         characters, on an optional basis
       - the user cannot force use of grouping if his/her locale is not configured for it
       - scanf functions may not understand the grouping character format string flag
    */

    // parse formatted temporary string and determine certain positions, such
    // as non-fractional digits
    const char * CharPtr = NumberString.c_str();
    int Index = 0;
    int DecimalSeparatorIndex = -1;
    int DigitStartIndex = -1;
    int DigitEndIndex = -1;
    while (*CharPtr != '\0')
      {
        char Char = *CharPtr;
        if (Char == NumberFormattingInternals -> m_DecimalSeparatorChar)
          {
            if (DecimalSeparatorIndex == -1)
              DecimalSeparatorIndex = Index;
          }

        if (wxIsdigit (Char))
          {
            if (DigitStartIndex == -1)
              {
                DigitStartIndex = Index;
              }
            if ((*(CharPtr + 1) == '\0')
                && DigitEndIndex == -1)
              {
                DigitEndIndex = Index + 1;
              }
          }
        else
          {
            if (DigitStartIndex != -1
                && DigitEndIndex == -1)
              {
                DigitEndIndex = Index;
              }
          }

        ++Index;
        ++CharPtr;
      }

    bool IsUseGrouping = NumberFormatOptions.m_GroupSize > 0;
    int GroupCharacterCount = 0;

    if (IsUseGrouping
        && (DigitStartIndex != -1) 
        && (DigitEndIndex != -1))
      {
        GroupCharacterCount = ((DigitEndIndex - DigitStartIndex) / NumberFormatOptions.m_GroupSize);
        if (GroupCharacterCount > 0)
          --GroupCharacterCount;
      }

    String.Clear();
    String.Alloc (NumberString.Length() + GroupCharacterCount);

    // now produce string with inserted grouping characters
    CharPtr = NumberString.c_str();
    Index = 0;
    while (*CharPtr != '\0')
      {
        if (IsUseGrouping
            && (DigitStartIndex != -1) 
            && (DigitEndIndex != -1)
            && (Index >= DigitStartIndex)
            && (Index < DigitEndIndex))
          {
            if ((Index < DigitEndIndex)
                && (Index > DigitStartIndex)
                && (((DigitEndIndex - Index) % NumberFormatOptions.m_GroupSize) == 0))
              {
                // insert grouping character
                String += NumberFormatOptions.m_GroupSeparatorChar;
              }
          }
        if (Index == DecimalSeparatorIndex)
          {
            // replace decimal separator
            String += NumberFormatOptions.m_DecimalSeparatorChar;
          }
        else
          {
            // otherwise add characters from temporary string
            String += *CharPtr;
          }

        ++Index;
        ++CharPtr;
      }
}

//-------------------------------------------------------------------------

#define M_CheckString(DesiredResult) \
    if (String != DesiredResult) \
      { \
        wxString MessageString; \
        MessageString << "string \"" << String << "\" obtained instead of desired result \"" << DesiredResult << "\""; \
        wxASSERT_MSG (false, MessageString);   \
        IsError = true; \
        break; \
      } \

bool testNumberFormat()
{
    wxString String;
    wxNumberFormatOptions NumberFormatOptions (2, '.', ';');

    bool IsOkay = false;
    bool IsError = false;
    for_once
      {
        // NOTE: different compilers/libraries may use differing 
        // interpretation of printf() format specifiers, causing M_CheckString
        // to indicate failure

        NumberFormatOptions.set (2, '.', ';');
        formatDouble ("%.1f", NumberFormatOptions, 12345678.9, String);
        M_CheckString ("12.34.56.78;9");

        NumberFormatOptions.set (2, '.', ';');
        formatDouble ("%g", NumberFormatOptions, 9, String);
        M_CheckString ("9");

        NumberFormatOptions.set (2, '.', ';');
        formatDouble ("%g", NumberFormatOptions, 9e50, String);
#if (defined(_MSC_VER)) // ?? || (defined(__BORLANDC__))
        M_CheckString ("9e+050");
#else
        M_CheckString ("9e+50");
#endif

        NumberFormatOptions.set (4, '.', ';');
        formatDouble ("%g", NumberFormatOptions, 23456789, String);
#if (defined(_MSC_VER)) // ?? || (defined(__BORLANDC__))
        M_CheckString ("2;34568e+007");
#else
        M_CheckString ("2;34568e+07");
#endif

        NumberFormatOptions.set (1, ' ', ';');
        formatDouble ("%.0f", NumberFormatOptions, -3456789, String);
        M_CheckString ("-3 4 5 6 7 8 9");

        NumberFormatOptions.set (4, '.', ';');
        formatDouble ("%g", NumberFormatOptions, 56789, String);
        M_CheckString ("5.6789");

        NumberFormatOptions.set (4, '.', ';');
        formatDouble ("%g", NumberFormatOptions, 6789, String);
        M_CheckString ("6789");

        // still, specified field width is modified because of inserting
        // grouping characters
        NumberFormatOptions.set (3, '.', ',');
        formatLong ("%14ld", NumberFormatOptions, -1123456789, String);
        M_CheckString ("   -1.123.456.789");

        NumberFormatOptions.set (3, '.', ',');
        formatLong ("%ld", NumberFormatOptions, 0, String);
        M_CheckString ("0");

        IsOkay = true;
      }

    return IsOkay;
}

#undef M_CheckString

//=========================================================================

wxDateTime::Month getMonthFromMonthIndex (int MonthIndex) // MonthIndex starting from 1, _not_ 0!
{
    switch (MonthIndex)
      {
        case 1: return wxDateTime::Jan;
        case 2: return wxDateTime::Feb;
        case 3: return wxDateTime::Mar;
        case 4: return wxDateTime::Apr;
        case 5: return wxDateTime::May;
        case 6: return wxDateTime::Jun;
        case 7: return wxDateTime::Jul;
        case 8: return wxDateTime::Aug;
        case 9: return wxDateTime::Sep;
        case 10: return wxDateTime::Oct;
        case 11: return wxDateTime::Nov;
        case 12: return wxDateTime::Dec;
      }
    return wxDateTime::Inv_Month;
}

int getMonthIndexFromMonth (wxDateTime::Month Month) // MonthIndex starting from 1, _not_ 0!
{
    switch (Month)
      {
        case wxDateTime::Jan: return 1;
        case wxDateTime::Feb: return 2;
        case wxDateTime::Mar: return 3;
        case wxDateTime::Apr: return 4;
        case wxDateTime::May: return 5;
        case wxDateTime::Jun: return 6;
        case wxDateTime::Jul: return 7;
        case wxDateTime::Aug: return 8;
        case wxDateTime::Sep: return 9;
        case wxDateTime::Oct: return 10;
        case wxDateTime::Nov: return 11;
        case wxDateTime::Dec: return 12;
      default:
        break;
      }
    return 0;
}

//=========================================================================

void getDecomposedDateFormatString (const wxString & InputFormatString,
                                    wxString & DecomposedDateFormatString)
{
    // our sample date: Monday, 22 October 1990
    // - the numbers should be unique (don't use 12 December or the like)
    // - the numbers should be 2-digit (so we don't have to care if a leading
    //   zero is printed or is not printed) - to simplify things a little
    wxDateTime SampleDateTime (/* day */ 22, 
                             /* month */ wxDateTime::Oct, 
                             /* year */ 1990);
    //  wxDateTime_t hour = 0, wxDateTime_t minute = 0,
    // wxDateTime_t second = 0, wxDateTime_t millisec = 0)    

    // WARN
    // - maybe it would be more appropriate to call CallStrftime() directly
    //   instead of calling wxDateTime::Format()
    // - should check if wxDateTime::Format()/CallStrftime() returned
    //   an error

    // get abbreviated and full weekday string for Monday for current locale
    wxString AbbreviatedSampleWeekdayString = SampleDateTime.Format ("%a"); 
    wxString FullSampleWeekdayString = SampleDateTime.Format ("%A");

    // get abbreviated and full month name for October for current locale
    wxString AbbreviatedSampleMonthString = SampleDateTime.Format ("%b"); 
    wxString FullSampleMonthString = SampleDateTime.Format ("%B");

    // format the sample date
    wxString SampleDateTimeString = SampleDateTime.Format (InputFormatString);

    // replace occurences of "%" with "%%" first:
    SampleDateTimeString.Replace ("%", "%%");

    // BUG: simply searching/replacing "%X" is incorrect if the format 
    // string contains the sequence "%%" (then, the "%a" in "%%a" is not
    // a placeholder) - to fix, add full parsing of format string
    // 
    SampleDateTimeString.Replace (FullSampleWeekdayString, "%A");
    SampleDateTimeString.Replace (AbbreviatedSampleWeekdayString, "%a");
    SampleDateTimeString.Replace ("22", "%d");
    SampleDateTimeString.Replace (FullSampleMonthString, "%B");
    SampleDateTimeString.Replace (AbbreviatedSampleMonthString, "%b");
    SampleDateTimeString.Replace ("10", "%m");
    SampleDateTimeString.Replace ("1990", "%Y");
    SampleDateTimeString.Replace ("90", "%y"); // maybe use %Y here too

    // now, SampleDateTimeString should contain a valid format string according
    // to the current locale setting
    DecomposedDateFormatString = SampleDateTimeString;

    // TODO: add checking if it was actually possible to determine
    // the decomposed format string; add a fall-back
}

void getDecomposedDateFormatStringX (wxString & DecomposedDateFormatString,
                                     bool IsLongFormat)
{
    if (IsLongFormat)
      {
#if defined(__WXMSW__)
#if (!defined(_MSC_VER)) && (!defined(__BORLANDC__))
#   pragma message ("Please check if your compiler/OS uses '%%#x' or '%%Ex' for alternative date format")
#endif
        // the format string to indicate alternative date/time 
        // representation differs in MSW/VC++, Borland C++
        // TODO check which other compilers use %#c instead of %Ec
        getDecomposedDateFormatString ("%#x", DecomposedDateFormatString);
#else
        getDecomposedDateFormatString ("%Ex", DecomposedDateFormatString);
#endif
      }
    else
      {
        getDecomposedDateFormatString ("%x", DecomposedDateFormatString);
      }
}

bool removeWeekdayFromDateFormatString (const wxString & DateFormatString,
                                        wxString & WithoutWeekdayDateFormatString)
{
    // make format string with weekday placeholder removed:
    // remove %a or %A, remove following spaces, punctuation characters etc.
    // WARN may have to remove spaces, punctuation characters before %a, %A as well
    //
    // BUG: simply searching for "%X" is incorrect if the format string
    // contains the sequence "%%" (then, the "%a" in "%%a" is not
    // a placeholder) - to fix, add full parsing of format string
    // 
    int WeekdayPlaceholderCharIndex = DateFormatString.Find ("%a");
    if (WeekdayPlaceholderCharIndex == wxNOT_FOUND)
      WeekdayPlaceholderCharIndex = DateFormatString.Find ("%A");
    if (WeekdayPlaceholderCharIndex != wxNOT_FOUND)
      {
        int CurrentCharIndex = WeekdayPlaceholderCharIndex + 2 /* strlen ("%a") */;
        for (; cast_is_smaller (CurrentCharIndex, DateFormatString.Length());
             ++CurrentCharIndex)
          {
            int CurrentChar = DateFormatString [CurrentCharIndex];
            if (!(wxIsspace (CurrentChar)
                  || (wxIspunct (CurrentChar) && CurrentChar != '%')))
              break;
          }
        WithoutWeekdayDateFormatString = DateFormatString;
        WithoutWeekdayDateFormatString.Remove (WeekdayPlaceholderCharIndex, 
                                               CurrentCharIndex-WeekdayPlaceholderCharIndex);

        return true;
      }
    return false;
}

//-------------------------------------------------------------------------

void formatDate (const wxDateTime & DateTime, wxString & String, 
                 bool IsLongFormat)
{
    // NOTE date formatting using "%#x" or "%Ex"
    // sometimes wxDateTime uses its own implementation which 
    // doesn't understand 'E' or '#'. The wxWindows implementation
    // is apparently used for dates before 2 Janary 1970
    // (an assertion failure is reported, and the output string
    // is set to the format string minus the percent character)
    // 
    // following hack is not required anymore, now there
    // is getDecomposedDateFormatString() which works better
    // 
    // if (TextCtrlContentString.IsSameAs ("#x")
    //  || TextCtrlContentString.IsSameAs ("Ex"))
    //   {
    //  // since we don't know if the current locale has 
    //  // DAY MONTH or MONTH DAY order, we output the date with the
    //  // month name to avoid any ambiguities
    //  TextCtrlContentString = TempDateTime.Format("%d %B %Y");
    // }

    wxString DateFormatString;
    getDecomposedDateFormatStringX (DateFormatString, IsLongFormat);
    
    String = DateTime.Format (DateFormatString);
}

//-------------------------------------------------------------------------

// parseDate()
// BUG
// - correct parsing of input (in case of occurences of "%%") would
//   be required instead of simply replacing %X, if any of the format
//   strings tried contains "%%"
//
bool parseDate (const wxString & String, wxDateTime & DateTime)
{
    // wxDateTime::ParseDate() works quite well unless the year in the string
    // is 2 digit and less than 31... Such a date string is generated
    // by Format("%#x") if no locale or some English locale was set 
    // (so the output is e.g. 02/19/04)
    // ParseDate() doesn't handle such a date.
    //
    // ParseFormat() may only be able to parse formats like dd/mm/yy or
    // mm/dd/yy, because possibly 'strptime' is not available. 
    // In this case, ParseFormat() cannot parse a date like '2 March 2002'.
    //
    // the following hack was used: The solution is to use try first 
    // ParseFormat() and if it fails try ParseDate(). 
    // However, this still doesn't work well because of the following problem:
    // 
    // possibly BUG in wxDateTime::ParseFormat():
    // for "%x", it uses "%d/%m/%y" as format if the time zone is e.g. European.
    // But the locale setting may well differ from the time zone setting.
    // (scope: wxWindows 2.4.1). 
    // Same problem may exist for ParseDate().
    //
    //!! A solution to this problem would be to make a dummy print of 
    //!! a fixed date like 10 November 2012 (2012-11-10) and then 
    //!! look in the output string if it was printed in day-month 
    //!! or month-day order
    //!! we could even try to construct a format string from the dummy
    //!! result
    // (the last idea is now implemented with getDecomposedDateFormatString())
    //
    // hack is obsolete now since we have getDecomposedDateFormatString()
    // which works well
    // --->
    // WARN see bug in ParseFormat() above
    // const char * EndCharPtr = TempDateTime.ParseFormat (BeginCharPtr, "%x");
    // bool IsOk = true;
    // if ((EndCharPtr == NULL) || (EndCharPtr == BeginCharPtr) || (*EndCharPtr != 0))
    //   {
    //  // try conversion with ParseDate() if ParseFormat() failed
    //  EndCharPtr = TempDateTime.ParseDate (BeginCharPtr);
    //  if ((EndCharPtr == NULL) || (EndCharPtr == BeginCharPtr) || (*EndCharPtr != 0))
    //    {
    //      IsOk = false;
    //    }
    //   }
    // <----

    const char * BeginCharPtr = String.c_str();
    const char * EndCharPtr = NULL;

    // try parsing the input as short date format:
    wxString ShortDateFormatString;
    getDecomposedDateFormatStringX (ShortDateFormatString, false);
    // note: under some platforms/locals, %x is equal to "%d.%m.%Y,
    // but we must parse with 2-digit year first because a 2-digit year
    // in the input may be interpreted as first-century year by
    // wxDateTime::ParseFormat().
    ShortDateFormatString.Replace ("%Y", "%y"); // make year 2-digit
    EndCharPtr = DateTime.ParseFormat (BeginCharPtr, ShortDateFormatString);
    if (!((EndCharPtr == NULL) || (EndCharPtr == BeginCharPtr) || (*EndCharPtr != 0)))
      {
        return true;
      }

    // try to parse input as short format, but with 4-digit year now
    ShortDateFormatString.Replace ("%y", "%Y");
      {
        EndCharPtr = DateTime.ParseFormat (BeginCharPtr, ShortDateFormatString);
        if (!((EndCharPtr == NULL) || (EndCharPtr == BeginCharPtr) || (*EndCharPtr != 0)))
          {
            return true;
          }
      }

    // try to parse input as long date format
    wxString DateFormatString;
    getDecomposedDateFormatStringX (DateFormatString, true);
    EndCharPtr = DateTime.ParseFormat (BeginCharPtr, DateFormatString);
    if (!((EndCharPtr == NULL) || (EndCharPtr == BeginCharPtr) || (*EndCharPtr != 0)))
      {
        return true;
      }

    // try to parse with weekday removed from long format string:
    wxString WithoutWeekdayDateFormatString;
    if (removeWeekdayFromDateFormatString (DateFormatString, 
                                           WithoutWeekdayDateFormatString))
      {
        const char * EndCharPtr = DateTime.ParseFormat (BeginCharPtr, 
                                                        WithoutWeekdayDateFormatString);
        if (!((EndCharPtr == NULL) || (EndCharPtr == BeginCharPtr) || (*EndCharPtr != 0)))
          {
            return true;
          }
      }

    // try to parse input date in ISO-8601 format (e.g. 2004-06-01)
    // WARN 
    // - Danish locale (and some others) may use %d-%m-%y as short date format, 
    //   so this may be ambiguous with ISO-8601 %Y-%m-%d
    // - at least, it would be wise not to use ISO-8601 with a two-digit year
    wxString ISO8601DateFormatString = "%Y-%m-%d";
    int DummyYear = 0;
    int ParsedYearDigits = 0;
    if ((wxSscanf (BeginCharPtr, "%d%n-", & DummyYear, & ParsedYearDigits) >= 1)
        && ParsedYearDigits == 4)
      {
        EndCharPtr = DateTime.ParseFormat (BeginCharPtr, ISO8601DateFormatString);
        if (!((EndCharPtr == NULL) || (EndCharPtr == BeginCharPtr) || (*EndCharPtr != 0)))
          {
            return true;
          }
      }

    return false;
}

//-------------------------------------------------------------------------

bool parseDateTimeWithFormat (const wxString & InputString, wxDateTime & DateTime,
                              const wxString & DateTimeFormatString)
{
    const char * BeginCharPtr = InputString.c_str();
    const char * EndCharPtr = NULL;
    {
      // use a different default date/time than default argument for
      // ParseFormat() to get more deterministic behavior (default would
      // be otherwise wxDateTime::Today())
      wxDateTime DefaultDateTime = wxDateTime (1, wxDateTime::Jan, 1970);

      EndCharPtr = DateTime.ParseFormat (BeginCharPtr, DateTimeFormatString,
                                         DefaultDateTime);
      if (!((EndCharPtr == NULL) || (EndCharPtr == BeginCharPtr) || (*EndCharPtr != 0)))
        {
          return true;
        }
    }
    return false;
}

void formatDateTimeWithFormat (const wxDateTime & DateTime, wxString & String,
                               const wxString & DateTimeFormatString)
{
    String = DateTime.Format (DateTimeFormatString);
}

//-------------------------------------------------------------------------

bool parseDateTimeFromISOFormat (const wxString & InputString, wxDateTime & DateTime)
{
    // try to parse input date in ISO-8601 format (e.g. 2004-06-01)
    // WARN
    // - Danish locale (and some others) may use %d-%m-%y as short date format,
    //   so this may be ambiguous with ISO-8601 %Y-%m-%d
    // - at least, it would be wise not to use ISO-8601 with a two-digit year
    wxString ISODateTimeFormatString = "%Y-%m-%d %H:%M:%S";
    return parseDateTimeWithFormat (InputString, DateTime, 
                                    ISODateTimeFormatString);
}

void formatDateTimeInISOFormat (const wxDateTime & DateTime, wxString & String)
{
    String = "";
    String 
      << DateTime.FormatISODate() << " " << DateTime.FormatISOTime();
}


//=========================================================================

void fitInside (const wxRect & FrameRect, const wxSize & ObjectSize, 
                wxRect & FittedObjectRect, int AlignmentFlags)
{
    int FrameRectWidth = FrameRect.GetWidth();
    int FrameRectHeight = FrameRect.GetHeight();
    int FittedX = FrameRect.GetX();
    int FittedY = FrameRect.GetY();
    int FittedWidth;
    int FittedHeight;
    double FrameRectWidthHeightRatio = (double) FrameRectWidth
      / (double) FrameRectHeight;
    double ObjectWidthHeightRatio = (double) ObjectSize.GetWidth()
      / (double) ObjectSize.GetHeight();

    if (FrameRectWidthHeightRatio > ObjectWidthHeightRatio)
      {
        cast_round_assign (FittedWidth, FrameRectHeight * ObjectWidthHeightRatio);
        FittedHeight = FrameRectHeight;

        if (AlignmentFlags & wxALIGN_CENTER_HORIZONTAL)
          {
            FittedX += (FrameRectWidth - FittedWidth) / 2;
          }
        else if (AlignmentFlags & wxALIGN_RIGHT)
          {
            FittedX += (FrameRectWidth - FittedWidth);
          }
      }
    else
      {
        FittedWidth = FrameRectWidth;
        cast_round_assign (FittedHeight, FrameRectWidth * 1/ObjectWidthHeightRatio);

        if (AlignmentFlags & wxALIGN_CENTER_VERTICAL)
          {
            FittedY += (FrameRectHeight - FittedHeight) / 2;
          }
        else if (AlignmentFlags & wxALIGN_BOTTOM)
          {
            FittedY += (FrameRectHeight - FittedHeight);
          }
      }

    FittedObjectRect = wxRect (FittedX, FittedY, FittedWidth, FittedHeight);
}

//=========================================================================

void makePath (const wxString & LhsPathString,
               const wxString & RhsPathString,
               wxString & ResultPathString,
               wxPathFormat LhsPathFormat,
               wxPathFormat RhsPathFormat,
               wxPathFormat ResultPathFormat)
{
    wxFileName FileName (RhsPathString, RhsPathFormat);
    // alternative could be to use wxFileName::AppendDir() or
    // wxFileName::PrependDir()
    FileName.MakeAbsolute (LhsPathString, LhsPathFormat);
    ResultPathString = FileName.GetFullPath (ResultPathFormat);
}

wxString makePath (const wxString & LhsPathString,
                   const wxString & RhsPathString,
                   wxPathFormat LhsPathFormat,
                   wxPathFormat RhsPathFormat,
                   wxPathFormat ResultPathFormat)
{
    wxString ResultString;
    makePath (LhsPathString, 
              RhsPathString,
              ResultString,
              LhsPathFormat,
              RhsPathFormat,
              ResultPathFormat);
    return ResultString;
}

//-------------------------------------------------------------------------

void makeAbsolutePath (const wxString & RelativeOrAbsolutePathString,
                       const wxString & ReferenceDirString,
                       wxString & AbsolutePathString,
                       wxPathFormat RelativeOrAbsolutePathFormat,
                       wxPathFormat ReferenceDirFormat,
                       wxPathFormat AbsolutePathFormat)
{
    wxFileName FileName (RelativeOrAbsolutePathString, RelativeOrAbsolutePathFormat);
    // alternative could be to use wxFileName::AppendDir() or
    // wxFileName::PrependDir()
    FileName.MakeAbsolute (ReferenceDirString, ReferenceDirFormat);
    AbsolutePathString = FileName.GetFullPath (AbsolutePathFormat);
}

wxString makeAbsolutePath (const wxString & RelativeOrAbsolutePathString,
                           const wxString & ReferenceDirString,
                           wxPathFormat RelativeOrAbsolutePathFormat,
                           wxPathFormat ReferenceDirFormat,
                           wxPathFormat AbsolutePathFormat)
{
    wxString AbsolutePathString;
    makeAbsolutePath (RelativeOrAbsolutePathString,
                      ReferenceDirString,
                      AbsolutePathString,
                      RelativeOrAbsolutePathFormat,
                      ReferenceDirFormat,
                      AbsolutePathFormat);
    return AbsolutePathString;
}

//-------------------------------------------------------------------------

void makeRelativePath (const wxString & RelativeOrAbsolutePathString,
                       const wxString & ReferenceDirString,
                       wxString & RelativePathString,
                       bool IsUseRelativeParentDir,
                       wxPathFormat RelativeOrAbsolutePathFormat,
                       wxPathFormat ReferenceDirFormat,
                       wxPathFormat RelativePathFormat)
{
    wxFileName FileName (RelativeOrAbsolutePathString, RelativeOrAbsolutePathFormat);
    FileName.MakeRelativeTo (ReferenceDirString, ReferenceDirFormat);

    if ((!IsUseRelativeParentDir)
        && (FileName.GetFullPath().StartsWith ("..")))
      {
        RelativePathString = RelativeOrAbsolutePathString;
      }
    else
      {
        RelativePathString = FileName.GetFullPath (RelativePathFormat);
      }
}

wxString makeRelativePath (const wxString & RelativeOrAbsolutePathString,
                           const wxString & ReferenceDirString,
                           bool IsUseRelativeParentDir,
                           wxPathFormat RelativeOrAbsolutePathFormat,
                           wxPathFormat ReferenceDirFormat,
                           wxPathFormat RelativePathFormat)
{
    wxString RelativePathString;
    makeRelativePath (RelativeOrAbsolutePathString,
                      ReferenceDirString,
                      RelativePathString,
                      IsUseRelativeParentDir,
                      RelativeOrAbsolutePathFormat,
                      ReferenceDirFormat,
                      RelativePathFormat);
    return RelativePathString;
}

//-------------------------------------------------------------------------

void addConcatenatedStrings (const wxArrayString & StringArray, 
                             wxString & OutputString,
                             const wxString & ElementPrefixString,
                             const wxString & ElementPostfixString,
                             const wxString & InfixString)
{
    int Count = StringArray.GetCount();
    bool IsElementPrefix = !ElementPrefixString.IsEmpty();
    bool IsElementPostfix = !ElementPostfixString.IsEmpty();
    // bool IsInfix = !InfixString.IsEmpty();
    for (int Index = 0; Index < Count; ++Index)
      {
        if (Index > 0)
          OutputString << InfixString;

        if (IsElementPrefix)
          OutputString << ElementPrefixString;

        OutputString << StringArray.Item(Index);

        if (IsElementPostfix)
          OutputString << ElementPostfixString;
      }
}

//-------------------------------------------------------------------------

const char * wxGetIdTranslation(const char * MessageIdentCharPtr, const char * DefaultCharPtr)
{
    const char * TranslatedCharPtr = wxGetTranslation (MessageIdentCharPtr);
    const char CheckedPrefixCharArray [] = "{id:";
    // TODO: could replace strncmp() with string pointer comparison to detect
    // if a translation is available
    if (strncmp (TranslatedCharPtr, CheckedPrefixCharArray, 
                 WXSIZEOF(CheckedPrefixCharArray) - 1) == 0)
      {
        return DefaultCharPtr;
      }
    else
      {
        return TranslatedCharPtr;
      }
}

//-------------------------------------------------------------------------

const char * wxGetCatalogMetaData(const char * MessageIdentCharPtr)
{
    const char * TranslatedCharPtr = wxGetTranslation (MessageIdentCharPtr);
    if (TranslatedCharPtr == MessageIdentCharPtr)
      {
        return NULL;
      }
    else
      {
        return TranslatedCharPtr;
      }
}

//-------------------------------------------------------------------------

wxCatalogAddManag::wxCatalogAddManag()
{
    init();
}

void wxCatalogAddManag::init ()
{
    m_IsError = false;
}

void wxCatalogAddManag::setLocale (wxLocale * Locale)
{
    m_Locale = Locale;
}

bool wxCatalogAddManag::addCatalog (const wxString & CatalogName)
{
    if (m_Locale -> GetLanguage() == wxLANGUAGE_UNKNOWN)
      {
        // don't try to load catalogue if language is not set or known
      }
    else 
      {
        if (!m_Locale -> AddCatalog (CatalogName))
          {
            m_IsError = true;
            // m_MessageString << "failed to load catalog \"" << CatalogName << "\".\n";
            m_MessageString << CatalogName << ":\n" << "    failed to load catalogue\n";
            return false;
          }
        // else
          // m_MessageString << CatalogName << ": loaded\n";
      }

    return true;
}

bool wxCatalogAddManag::addCatalogWithVersionCheck (const wxString & CatalogName,
                                                    const wxString & RequiredVersionString)
{
    if (m_Locale -> GetLanguage() == wxLANGUAGE_UNKNOWN)
      {
        // don't try to load catalogue if language is not set or known
      }
    else if (addCatalog (CatalogName))
      {
        wxString VersionMessageIdentString = wxString("{meta:") + CatalogName + ":VersionIdent}";
        const char * ActualVersionString = wxGetCatalogMetaData (VersionMessageIdentString);

        if ((ActualVersionString == NULL)
            || (strcmp (ActualVersionString, RequiredVersionString) != 0))
          {
            m_MessageString
              << CatalogName << ":\n"
              << "    required version: " << RequiredVersionString << "\n"
              << "    actual version: " << ((ActualVersionString != NULL) ? ActualVersionString : "(unknown)")
              << "\n";

            m_IsError = true;
          }

        return false;
      }

    return true;
}

bool wxCatalogAddManag::getMessage (wxString & MessageString)
{
    if (true)
      {
        int LanguageId = m_Locale -> GetLanguage();
        const struct wxLanguageInfo * LanguageInfoPtr = wxLocale::GetLanguageInfo (LanguageId);
        MessageString << "Language: ";
        if (LanguageInfoPtr != NULL)
          {
            MessageString
              << LanguageInfoPtr->Description << " (" << LanguageInfoPtr->CanonicalName << ")\n\n";
          }
        else
          {
            MessageString << "(unknown)" << "\n\n";
          }
      }

    if (m_IsError)
      {
        MessageString
          << "Some of the message translation catalogues could not be loaded "
          << "or do not have the expected versions:\n\n";
      }

    // output messages about specific language versions
    MessageString << m_MessageString;

    return (m_IsError);
}


#if 0
extern bool checkCatalogueMetaData (wxString & MessageString,
                                    const char * CatalogueNameString,
                                    const char * VersionMessageIdentCharPtr,
                                    const char * RequiredVersionString)
{
    bool IsOkay = false;
    const char * ActualVersionString = wxGetCatalogueMetaData (VersionMessageIdentCharPtr);

#if 0    
    MessageString
      << VersionMessageIdentCharPtr << ":\n"
      << "    required value: " << RequiredVersionString << "\n"
      << "    actual value: " << ((ActualVersionString != NULL) ? ActualVersionString : "(not found)") << "";
#endif
    MessageString
      << CatalogueNameString << ":\n"
      << "    required version: " << RequiredVersionString << "\n"
      << "    actual version: " << ((ActualVersionString != NULL) ? ActualVersionString : "(not found)") << "";
    
    if (ActualVersionString == NULL)
      {
#if 0
        MessageString
          << "could not find entry for meta-data key \"" << VersionMessageIdentCharPtr << "\" "
          << "in message catalogue \"" << CatalogueNameString << "\", "
          << "which must match \"" << RequiredVersionString << "\" ";
#endif
      }
    else
      {
        if (strcmp (ActualVersionString, RequiredVersionString) != 0)
          {
#if 0
            MessageString
              << "entry for meta-data \"" << VersionMessageIdentCharPtr << "\" "
              << "in message catalogue " << CatalogueNameString << ", "
              << "which must match \"" << RequiredVersionString << "\" "
              << "has a value of \"" << ActualVersionString << "\" ";
#endif
          }
        else
          {
#if 0
            MessageString
              << "entry for meta-data \"" << VersionMessageIdentCharPtr << "\" "
              << "in message catalogue " << CatalogueNameString << ", "
              << "matches required value of \"" << RequiredVersionString << "\" ";
#endif
            IsOkay = true;
          }
      }
    MessageString << "\n\n";
    return IsOkay;
}
#endif

//=========================================================================

#if defined(__WXMSW__)

// // wxGetInstance() is somehow needed for loadDataResource():
// // (as declared in ./include/wx/msw/private.h):
// WXDLLEXPORT HINSTANCE wxGetInstance();
// NOTE we use GetModuleHandle(NULL) instead of wxGetInstance()
// which may not work under Win16

// NOTE semantics not well defined for Win16: there, some function like
// releaseDataResource() or copying the data block would be needed
bool loadDataResource (const wxString & ResourceNameString, 
                       const char * ResourceTypeString,
                       unsigned char ** DataPtr, size_t * DataSize)
{
    bool IsCompleted = false;

    for_once
      {
        HRSRC ResourceHandle = ::FindResource (::GetModuleHandle(NULL) /*wxGetInstance()*/, 
                                               ResourceNameString, ResourceTypeString);
        if (ResourceHandle == 0)
          break;

        HGLOBAL DataHandle = NULL;
        if (DataPtr != NULL)
          {
            // load and lock data block (if desired, i.e. if DataPtr != NULL)
            DataHandle = ::LoadResource (::GetModuleHandle(NULL) /*wxGetInstance()*/, 
                                         ResourceHandle);
            if (DataHandle == 0)
              break;

            *DataPtr = (unsigned char *) ::LockResource (DataHandle);
            if (*DataPtr == NULL)
              break;
          }
            
        if (DataSize != NULL)
          {
            // determine size of the data block (if desired)
            *DataSize = ::SizeofResource (GetModuleHandle (NULL) /*wxGetInstance()*/, 
                                          ResourceHandle);
          }

        // BUG for Win16:
        // UnlockResource() must be called for Win16, however; we must
        // probably do this after working with the data

        IsCompleted = true;
      }
    
    return IsCompleted;
}

// void releaseDataResource ()
// {
//      if (DataPtr != NULL)
//        {
//          // UnlockResource() not needed for WIN32
// #ifndef __WIN32__
//          UnlockResource(DataHandle);
// #endif
//        }
    
//      // needed for Win16?
//      //  GlobalFree(hData);
// }

#endif // defined(__WXMSW__)

//=========================================================================

// bool loadFile (const wxString & FileNameString, 
//                wxByteArray & ByteArray)
// {
//     ByteArray.Clear();
//     ByteArray
//     wxFileInputStream FileInputStream = wxFileInputStream (FilenameString);
// 
//     if (!(FileInputStream -> Ok()))
//       return false;
// 
//     FileInputStream.Read (ByteArray, ByteCount);
// 
// }

//-------------------------------------------------------------------------

// WARN why conditional compilation only for wxGUI here?
#if wxUSE_GUI
bool loadImageFromMemory (const unsigned char * ImageData, size_t ImageDataSize,
                          wxImage & Image)
{
    wxMemoryInputStream MemoryInputStream (ImageData, ImageDataSize);
    return Image.LoadFile (MemoryInputStream);
}

bool loadBitmapFromMemory (const unsigned char * ImageData, size_t ImageDataSize,
                           wxBitmap ** Bitmap)
{
    *Bitmap = NULL;

    bool IsOk = false;
    for_once
      {
        wxImage Image;
        if (! loadImageFromMemory (ImageData, ImageDataSize, Image))
          break;

        *Bitmap = new wxBitmap (Image);
        if (*Bitmap == NULL)
          break;

        IsOk = true;
      }

    return IsOk;
}
#endif // wxUSE_GUI

#if defined(__WXMSW__)

#if wxUSE_GUI
bool loadImageFromResource (const wxString & ImageDataResourceNameString, 
                            wxImage & Image)
{
    unsigned char * DataPtr = NULL;
    size_t DataSize = 0;
    if (!loadDataResource (ImageDataResourceNameString, RT_RCDATA,
                           & DataPtr, & DataSize))
      return false;

    return loadImageFromMemory (DataPtr, DataSize, Image);
}

bool loadBitmapFromResource (const wxString & ImageDataResourceNameString, 
                             wxBitmap ** Bitmap)
{
    bool IsOk = false;

    *Bitmap = NULL;

    for_once
      {
        unsigned char * DataPtr = NULL;
        size_t DataSize = 0;
        if (!loadDataResource (ImageDataResourceNameString, RT_RCDATA,
                               & DataPtr, & DataSize))
          break;
    
        if (!loadBitmapFromMemory (DataPtr, DataSize, Bitmap))
          break;

        IsOk = true;
      }

    if (!IsOk)
      *Bitmap = new wxBitmap ();

    return IsOk;
}
#endif // wxUSE_GUI

#endif // defined(__WXMSW__)

//=========================================================================

#if (M_WxExtLib_IsUseGraphicsHelper == 1)
#  if wxUSE_GUI

// resizeImageVGrid()
// Todo:
// - support wxImage having alpha/transparency channel (wxWidgets 
//   version approx. >= 2.5)
// - check version numbers wxImage/wxCHECK_VERSION switching
bool resizeImageVGrid (const wxImage & InputImage,
                       wxImage & OutputImage,
                       int OutputWidth, int OutputHeight)
{
    if (!InputImage.Ok())
      return false;

#if wxCHECK_VERSION (2, 5, 0)
    if (!OutputImage.Create (OutputWidth, OutputHeight, false /* IsClear */))
      return false;
#else
    // overload resolution for older version may accidently interpret 'false' 
    // as pointer and so use Create() with initialization of allocated bytes:
    OutputImage.Create (OutputWidth, OutputHeight);
#endif

    if (!OutputImage.Ok())
      return false;

    const unsigned char * InputImageData = InputImage.GetData();
    unsigned char * OutputImageData = OutputImage.GetData();

#if wxCHECK_VERSION (2, 5, 0)
    if (InputImage.HasAlpha())
      return false;
#endif

	if (!resizeImageVGrid (InputImageData, InputImage.GetWidth() * 3,
                           InputImage.GetWidth(), InputImage.GetHeight(),
                           true /* IsRGB */ , false /* IsAlpha */,
                           OutputImageData, OutputImage.GetWidth() * 3,
                           OutputImage.GetWidth(), OutputImage.GetHeight()))
      return false;

    return true;
}
#  endif
#endif

//=========================================================================

// doTextLineWrap()
// - text extents are calculated for the font currently set for the DC
// Possible improvements:
// - Insert e.g. '\r\n' or '\r\r\n' when having a 'soft' line break (not forced)
//   instead of just '\n'. This way, the soft line breaks could be un-done
//   again (because then they would be distinguishable from the hard line breaks).
//   This could be useful if a control want's to re-do line wrapping 
//   e.g. after a size change notification.
void doTextLineWrap (const wxString & InputString, 
                     int MaxLineWidth,
                     wxDC & DC,
                     wxString & ResultString)
{
    wxString CurrentLineString;
    // pre-allocate line-string to improve performance:
    CurrentLineString.Alloc (512);

    const char * CurrentCharPtr = InputString.c_str();
    // const char * LineStartCharPtr = CurrentCharPtr;
    const char * LastBreakCharPtr = NULL;

    int CurrentLineCharCount = 0;
    int LastBreakCharCount = 0;

    bool IsJustAfterWrap;
    bool IsAfterWrap = false;
    while (1)
      {
        IsJustAfterWrap = false;
        bool IsBreakAllowed = false;
        bool IsBreakForced = false;

        int CurrentChar = *CurrentCharPtr;

        if (IsAfterWrap)
          {
            if (!(CurrentChar == '\0'
                  || wxIsspace (CurrentChar))
                || CurrentChar == '\n')
              IsAfterWrap = false;
          }

        if (!IsAfterWrap)
          {
            if (CurrentChar == '\0'
                || wxIsspace (CurrentChar))
              IsBreakAllowed = true;

            if (CurrentChar == '\n'
                || CurrentChar == '\0')
              IsBreakForced = true;
          }

        if (IsBreakAllowed
            || IsBreakForced)
          {
            int Width = 0;
            int Height = 0;
            // NOTE there is also wxWindow::GetTextExtent, maybe 
            // one could optionally use this function instead of wxDC::GetTextExtent
            DC.GetTextExtent (CurrentLineString, & Width, & Height);
        
            bool IsOverfull = (Width > MaxLineWidth);

            if (IsOverfull
                || IsBreakForced)
              {
                const char * ActualBreakCharPtr = NULL;
                if (IsOverfull)
                  {
                    ActualBreakCharPtr = LastBreakCharPtr;
                    if (ActualBreakCharPtr == NULL)
                      ActualBreakCharPtr = CurrentCharPtr;
                  }
                else
                  {
                    ActualBreakCharPtr = CurrentCharPtr;
                  }

                if (IsOverfull
                    && LastBreakCharPtr != NULL)
                  {
                    CurrentLineString.Truncate (LastBreakCharCount);
                  }

                ResultString.Append (CurrentLineString);
                ResultString.Append ("\n");
                
                // set string to empty but retain allocated memory:
                CurrentLineString.Empty ();

                // reposition pointers
                // LineStartCharPtr = ActualBreakCharPtr;
                CurrentCharPtr = ActualBreakCharPtr;
                CurrentChar = *CurrentCharPtr;
                LastBreakCharPtr = NULL;
                LastBreakCharCount = 0;
                CurrentLineCharCount = 0;

                if (CurrentChar == '\n')
                  {
                    ++CurrentCharPtr;
                    CurrentChar = *CurrentCharPtr;
                  }

                IsAfterWrap = true;
                IsJustAfterWrap = true;
              }
            else
              {
                LastBreakCharPtr = CurrentCharPtr;
                LastBreakCharCount = CurrentLineCharCount;
              }
          }

        if (CurrentChar == '\0')
          break;

        if (!IsAfterWrap)
          {
            CurrentLineString.Append ((char) CurrentChar);
            ++CurrentLineCharCount;
          }

        if (!IsJustAfterWrap)
          {
            ++CurrentCharPtr;
          }
      }
}

//-------------------------------------------------------------------------

void doTextLineWrap (const wxString & InputString, 
                     int MaxLineWidth,
                     wxWindow * Window,
                     wxString & ResultString)
{
   wxClientDC ClientDC (Window);
   const wxFont & Font = Window -> GetFont ();

   ClientDC.SetFont (Font);
   doTextLineWrap (InputString, MaxLineWidth, ClientDC, ResultString);

   // WARN don't know if resetting old font is required
}

//=========================================================================

bool wxVerifyWindowPtr (wxWindow * ParentWindow, wxWindow * SearchedWindow)
{
    wxWindowListNode * Node = NULL;

    if (ParentWindow == SearchedWindow)
      return true;

    // search list of windows recursively, starting from specified
    // ParentWindow, depth-first search
    for (Node = ParentWindow->GetChildren().GetFirst();
         Node != NULL;
         Node = Node->GetNext())
      {
        wxWindow * Window = Node->GetData();
        bool IsFound = wxVerifyWindowPtr (Window, SearchedWindow);
        if (IsFound)
          return true;
      }

    return false;
}

//=========================================================================

wxFocusRestorer::wxFocusRestorer ()
{
    m_OldFocusWindow = wxWindow::FindFocus ();
    
    // save HWND?
    // HWND m_OldFocusHwnd = OldFocusWindow != NULL ? GetHwndOf (OldFocusWindow) : NULL;
}

wxFocusRestorer::~wxFocusRestorer ()
{
    if (m_OldFocusWindow != NULL)
      {
        wxWindow * CurrentFocusWindow = wxWindow::FindFocus ();

        // TEST: set focus window only if it appears to have changed
        // to avoid flicker (if this is the reason for flicker, maybe
        // it isn't)
        if (CurrentFocusWindow != m_OldFocusWindow)
          m_OldFocusWindow -> SetFocus();
      }
}

//=========================================================================

#if wxUSE_GUI
wxMultiLineText::wxMultiLineText ()
{
    m_TextCtrl = NULL;
    m_HtmlWindow = NULL;
    m_StaticText = NULL;

    m_MultiLineTextStyle = 0;
}

void wxMultiLineText::create (wxWindow * ParentWindow,
                              int WindowIdent,
                              int CtrlStyle,
                              const wxSize & MinSize,
                              int MultiLineTextStyle)
{
    wxSize UsedSize = MinSize;

    m_MultiLineTextStyle = MultiLineTextStyle;

    // determine what kind of borders to use:
    if (! (CtrlStyle & wxBORDER_NONE)
        && !(CtrlStyle & wxBORDER_SIMPLE))
      {
        // default is wxSUNKEN_BORDER:
        CtrlStyle |= wxSUNKEN_BORDER;
      }

    // determine if wxTextCtrl or wxHtmlWindow should be used:
    bool IsCreateTextCtrl = true;

#if ((defined __WXMOTIF__) || defined(__WXGTK__))
    // with Lesstif/Motif?, we don't want to use wxTextCtrl 
    // because it uses a fixed-size typewriter-like font 
    // and setting colours is not possible
    //
    // GTK (1.2.7) wxTextCtrl ignores setting foreground, background 
    // colours or not using sunken border
    // 
    // default for Motif/Lesstif, GTK: use wxHtmlWindow
    IsCreateTextCtrl = false;
#endif

    if (MultiLineTextStyle & wxFORCE_HTMLWINDOW)
      IsCreateTextCtrl = false;

    if (MultiLineTextStyle & wxFORCE_TEXTCTRL)
      IsCreateTextCtrl = true;

    if (MultiLineTextStyle & wxHTML_MODE)
      IsCreateTextCtrl = false;

    // create the desired control type
    if (IsCreateTextCtrl)
      {
        // borders: wxTextCtrl seems to normally have a border.
        // No border can be requested with wxBORDER_NONE.
        
        long TextCtrlStyle = wxTE_MULTILINE | wxTE_LINEWRAP | wxTE_READONLY;
        /* | wxVSCROLL | wxTE_RICH | wxBORDER_NONE */
        // if (! (CtrlStyle & wxSUNKEN_BORDER))
        //  TextCtrlStyle |= wxBORDER_NONE;
        TextCtrlStyle |= CtrlStyle;
        
        m_TextCtrl = new wxTextCtrl (ParentWindow, WindowIdent, _T(""),
                                     wxDefaultPosition, UsedSize,
                                     TextCtrlStyle,
                                     wxDefaultValidator);

        m_TextCtrl -> SetBackgroundColour (ParentWindow -> GetBackgroundColour());
      }
    else
      {
        // borders: wxHtmlWindow seems to support
        // wxSIMPLE_BORDER, wxSUNKEN_BORDER, (no border if no flag specified),
        // maybe others
        m_HtmlWindow = new wxHtmlWindow (ParentWindow, WindowIdent, 
                                         wxDefaultPosition, 
                                         UsedSize,
                                         CtrlStyle);
        m_HtmlWindow -> SetBorders (2);
        m_HtmlWindow -> SetBackgroundColour (ParentWindow -> GetBackgroundColour());
      }

//  // wxStaticText apparently ignores wxVSCROLL (at least under
//  // this X11 and this Lesstif), so it isn't very useful for our cause
//  // of displaying, possibly very long, multi-line text
//     m_StaticText = new wxStaticText (ParentWindow, WindowIdent, _T(""),
//                                   wxDefaultPosition, UsedSize,
//                                   /* WindowStyle */ wxVSCROLL);
    
}

//-------------------------------------------------------------------------

void wxMultiLineText::setText (const wxString & String)
{
    if (m_TextCtrl != NULL)
      {
#if defined(__WXMSW__)
        // the RichEdit supports line wrapping
        m_TextCtrl -> SetValue (String);
#else
        // under X11, the wxTextCtrl doesn't support wrapping of long
        // lines by itself, here we call doTextLineWrap()
        // (note: from wxWindows 2.6 (2.5.x) on, the wxX11/wxUniv wxTextCtrl 
        // will have support for line wrap)
        //
        // GTK 1.x: the wxTextCtrl supports wxTE_LINEWRAP, but
        // wxTextCtrl::GetNumberOfLines() only counts newlines
        // (but we are interessted in the number of lines after 
        // doing the wrapping) - because of this problem, we use
        // doTextLineWrap() for GTK as well
        // 
        wxSize ClientSize = m_TextCtrl -> GetClientSize();
        wxString ResultString;

        int MaxLineWidth = ClientSize.GetWidth();
        // HACK quick hack to account for possibly existing 
        // scrollbars:
        if (MaxLineWidth >= 25)
          MaxLineWidth -= 25;

#if defined(__WXMOTIF__)
        // the Lesstif wxTextCtrl has lot's of bordering around the
        // space actually usable for text, so subtract a little from
        // the horizontal size
        MaxLineWidth -= 20;
#endif

        // NOTE: the wxUniversal wxTextCtrl now (version 2.5.2 tested) has support 
        // for wrapping long lines, so calling doTextLineWrap() may not be required 
        // for this newer wxWindows versions
        // Also, this new wrapping mechanism has support for partial re-wrapping,
        // which is not available with doTextLineWrap().
        // (however, must check semantics of GetNumberOfLines())
        doTextLineWrap (String, MaxLineWidth, 
                        m_TextCtrl,
                        ResultString);
        m_TextCtrl -> SetValue (ResultString);
#endif
      }
    else if (m_HtmlWindow != NULL)
      {
        // WARN should add charset name in HTML <head> for currently used charset
        wxString HtmlString = "<html><body bgcolor=\"" 
          + formatColourToHtmlParam (m_HtmlWindow -> GetBackgroundColour())
          + "\" text=\""
          + formatColourToHtmlParam (m_HtmlWindow -> GetForegroundColour())
          + "\">\n"
          + "";
        if (m_MultiLineTextStyle & wxHTML_MODE)
          {
            HtmlString += String;
          }
        else
          {
            HtmlString += convertAsciiToHtml (String, true, 
                                              wxAsciiToHtml_ConvertLeadingSpaces);
          }
        HtmlString += "</body></html>";
        m_HtmlWindow -> SetPage (HtmlString);
      }
    else if (m_StaticText != NULL)
      {
        m_StaticText -> SetLabel (String);
      }
}

//-------------------------------------------------------------------------

wxWindow * wxMultiLineText::getWindow ()
{
   if (m_TextCtrl != NULL)
     return m_TextCtrl;
   else if (m_HtmlWindow != NULL)
     return m_HtmlWindow;
   else if (m_StaticText != NULL)
     return m_StaticText;
   else
     return NULL;
}

wxTextCtrl * wxMultiLineText::getTextCtrl ()
{
    return m_TextCtrl;
}

wxHtmlWindow * wxMultiLineText::getHtmlWindow ()
{
    return m_HtmlWindow;
}

wxStaticText * wxMultiLineText::getStaticText ()
{
    return m_StaticText;
}

//-------------------------------------------------------------------------

void wxMultiLineText::setTextCtrl (wxTextCtrl * TextCtrl, int MultiLineTextStyle)
{
    m_TextCtrl = TextCtrl;
    m_StaticText = NULL;
    m_HtmlWindow = NULL;

    m_MultiLineTextStyle = MultiLineTextStyle;
}

void wxMultiLineText::getStaticText (wxStaticText * StaticText, int MultiLineTextStyle)
{
    m_TextCtrl = NULL;
    m_StaticText = StaticText;
    m_HtmlWindow = NULL;

    m_MultiLineTextStyle = MultiLineTextStyle;
}

void wxMultiLineText::setHtmlWindow (wxHtmlWindow * HtmlWindow, int MultiLineTextStyle)
{
    m_TextCtrl = NULL;
    m_StaticText = NULL;
    m_HtmlWindow = HtmlWindow;

    m_MultiLineTextStyle = MultiLineTextStyle;
}
#endif // wxUSE_GUI

//=========================================================================

#if wxUSE_GUI

IMPLEMENT_DYNAMIC_CLASS(wxTransparentStaticText, wxStaticText)

BEGIN_EVENT_TABLE (wxTransparentStaticText, wxStaticText)
#if defined (__WXMSW__)
    EVT_ERASE_BACKGROUND (wxTransparentStaticText::OnEraseBackground)
    EVT_PAINT(wxTransparentStaticText::OnPaint)
#else
    // see notes below regarding transparency support in other ports
    // than wxMSW
#endif
END_EVENT_TABLE()

//-------------------------------------------------------------------------

wxTransparentStaticText::wxTransparentStaticText ()
{
}

bool wxTransparentStaticText::Create(wxWindow * ParentWindow,
                                     wxWindowID ControlIdent,
                                     const wxString& LabelString,
                                     const wxPoint& Position,
                                     const wxSize& Size,
                                     long Style,
                                     ETransparentStaticTextFlags Flags,
                                     const wxString& Name)
{
    m_TransparentStaticTextFlags = Flags;

    bool IsOk = wxStaticText::Create (ParentWindow, ControlIdent, LabelString,
                                      Position, Size, Style, Name);

    return IsOk;
}

//-------------------------------------------------------------------------

void wxTransparentStaticText::OnEraseBackground (wxEraseEvent & WXUNUSED(EraseEvent))
{
    // background is painted in OnPaint(), to avoid flicker

    // doesn't exist:
    // wxWindow::OnEraseBackground (EraseEvent);
}

void wxTransparentStaticText::OnPaint (wxPaintEvent & PaintEvent)
{
#if defined(__WXMSW__)
    // wxMSW port: wxStaticText doesn't seem to support text in transparency
    // mode, so implement our own paint function

    wxPaintDC PaintWxDC (this);

    wxRect WindowRect = GetClientRect();

    if (m_TransparentStaticTextFlags & IsOpaque)
      {
        // wxDC & DC = * EraseEvent.GetDC();

        PaintWxDC.SetBrush (wxBrush (GetBackgroundColour(), wxSOLID));
        PaintWxDC.SetPen (* wxTRANSPARENT_PEN);
        PaintWxDC.DrawRectangle (WindowRect);
      }

    // other settings which might be important for drawing:
    // - SetMapMode(), SetUserScale()
    // - SetTextForeground()
    // - SetTextBackground()
    // - SetClippingRegion()

    // PaintWxDC.SetPen (m_Pen);
    // PaintWxDC.SetBrush (m_Brush);
    // PaintWxDC.DrawRectangle (WindowRect);
    // PaintWxDC.DrawRoundedRectangle (WindowRect, -4.0);

    // WARN required to set these values back?
    PaintWxDC.SetBackgroundMode (wxTRANSPARENT);
    PaintWxDC.SetTextForeground (GetForegroundColour ());
    PaintWxDC.SetTextBackground (GetBackgroundColour ());
    PaintWxDC.SetFont (GetFont ());

    // DC::DrawText() seems to ignore ampersand characters, so use
    // Win32 API DrawText() instead
    // PaintWxDC.DrawText (wxGetWindowText(GetHWND()), WindowRect.GetX(), WindowRect.GetY());
    wxString WindowTextString = GetLabel();

    // const char * WindowTextCharPtr = WindowTextString.c_str();
    int CharCount = wxStrlen (WindowTextString);

    // GetPixel returns a COLORREF in Win32
    COLORREF ForegroundColorRef = GetForegroundColour().GetPixel();

    HDC PaintDC = (HDC) PaintWxDC.GetHDC();
    COLORREF PreviousForegroundColorRef = SetTextColor (PaintDC, ForegroundColorRef);
    int PreviousBkMode = SetBkMode (PaintDC, TRANSPARENT);
 
    // BUG TODO 
    // HDC SetFont
    // test/compile with other wx ports

    RECT Rect = {WindowRect.GetX(), WindowRect.GetY(), 
                 WindowRect.GetWidth(), WindowRect.GetHeight() };
    // TODO consider other DT_* flags
    ::DrawText (PaintDC, WindowTextString,
                CharCount, & Rect, DT_SINGLELINE /* | DT_CENTER */);

    SetBkMode (PaintDC, PreviousBkMode);
    SetTextColor (PaintDC, PreviousForegroundColorRef);
#else

#ifndef __WXGTK__
    // except for wxGTK, support for transparent text-drawing for the
    // different ports (wxMOTIF, wxUniversal etc.) in unkown/untested,
    // so add a warning here, at least
    // sure, above code for wxMSW can't be used in these ports anyway,
    // since it uses the Win32 API

    #pragma message ("Please check transparency support of wxStaticText in this wx-port");
#endif
    // ???
    // for wxGTK, neither wxStaticText nor wxWindow implements a method
    // OnPaint we could call, so resort to simply not defining OnPaint()
    // in wxGTK (and other ports), thus using the default implementation
    // of OnPaint() (see above for message handler definitions)
    // wxStaticText::OnPaint (PaintEvent);
    // wxWindow::OnPaint (PaintEvent);

#if wxCHECK_VERSION (2, 5, 3)
    wxUnusedVar (PaintEvent);
#endif
#endif
}
#endif // wxUSE_GUI

//=========================================================================

#if wxUSE_GUI

IMPLEMENT_DYNAMIC_CLASS(wxColourBarWindow, wxPanel)

BEGIN_EVENT_TABLE (wxColourBarWindow, wxPanel)
    EVT_ERASE_BACKGROUND (wxColourBarWindow::OnEraseBackground)
    EVT_PAINT(wxColourBarWindow::OnPaint)
END_EVENT_TABLE()

//-------------------------------------------------------------------------

wxColourBarWindow::wxColourBarWindow ()
{
    m_Brush = *wxTRANSPARENT_BRUSH;
    m_Pen = *wxTRANSPARENT_PEN;
    m_RoundedCornerRadius = 0.;
}

wxColourBarWindow::wxColourBarWindow (wxWindow * ParentWindow, int ControlIdent,
                                      const wxPoint & Position, 
                                      const wxSize & Size,
                                      long Style, const wxString& Name)
  : wxPanel (ParentWindow, ControlIdent, 
              Position, Size,
              Style, Name)
{
    m_Brush = *wxTRANSPARENT_BRUSH;
    m_Pen = *wxTRANSPARENT_PEN;
    m_RoundedCornerRadius = 0.;
}

wxColourBarWindow::wxColourBarWindow (wxWindow * ParentWindow, int ControlIdent,
                                      const wxColour & Colour,
                                      const wxPoint & Position, 
                                      const wxSize & Size,
                                      long Style, const wxString& Name)
  : wxPanel (ParentWindow, ControlIdent, 
              Position, Size,
              Style, Name)
{
    m_Brush.SetColour (Colour);
    m_Brush.SetStyle (wxSOLID);
    m_RoundedCornerRadius = 0.;
    m_Pen = *wxTRANSPARENT_PEN;
}

//-------------------------------------------------------------------------

void wxColourBarWindow::setBrush (const wxBrush & Brush)
{
    m_Brush = Brush;
}

void wxColourBarWindow::setRoundedCornerRadius (double RoundedCornerRadius)
{
    m_RoundedCornerRadius = RoundedCornerRadius;
}

void wxColourBarWindow::setPen (const wxPen & Pen)
{
    m_Pen = Pen;
}

//-------------------------------------------------------------------------

void wxColourBarWindow::OnEraseBackground (wxEraseEvent & WXUNUSED(EraseEvent))
{
    // wxPanel::OnEraseBackground (EraseEvent);
}

void wxColourBarWindow::OnPaint (wxPaintEvent & WXUNUSED(PaintEvent))
{
    wxPaintDC PaintWxDC (this);

    wxRect WindowRect = GetClientRect();

    PaintWxDC.SetPen (m_Pen);
    PaintWxDC.SetBrush (m_Brush);

    if (m_RoundedCornerRadius == 0.)
      {
        PaintWxDC.DrawRectangle (WindowRect);
      }
    else
      {
        PaintWxDC.DrawRoundedRectangle (WindowRect, m_RoundedCornerRadius);
      }
}
#endif // wxUSE_GUI

//=========================================================================

#if (M_WxExtLib_IsUseGraphicsHelper == 1)
#   if wxUSE_GUI

IMPLEMENT_DYNAMIC_CLASS(wxScaledImageWindow, wxPanel)

BEGIN_EVENT_TABLE (wxScaledImageWindow, wxPanel)
    EVT_ERASE_BACKGROUND (wxScaledImageWindow::OnEraseBackground)
    EVT_PAINT(wxScaledImageWindow::OnPaint)
END_EVENT_TABLE()

//-------------------------------------------------------------------------

wxScaledImageWindow::wxScaledImageWindow ()
{
    // m_OriginalImage = wxNullImage;
    m_IsCacheOkay = false;
}

wxScaledImageWindow::wxScaledImageWindow (wxWindow * ParentWindow, int ControlIdent,
                                          const wxPoint & Position, 
                                          const wxSize & Size,
                                          long Style, const wxString& Name)
  : wxPanel (ParentWindow, ControlIdent, 
             Position, Size,
             Style, Name)
{
    // m_OriginalImage = wxNullImage;
    m_IsCacheOkay = false;
}

wxScaledImageWindow::wxScaledImageWindow (wxWindow * ParentWindow, int ControlIdent,
                                          const wxImage & Image,
                                          const wxPoint & Position, 
                                          const wxSize & Size,
                                          long Style, const wxString& Name)
  : wxPanel (ParentWindow, ControlIdent, 
             Position, Size,
             Style, Name)
{
    m_OriginalImage = Image;
    m_IsCacheOkay = false;
}

//-------------------------------------------------------------------------

void wxScaledImageWindow::setImage (const wxImage & Image)
{
    m_OriginalImage = Image;
    m_IsCacheOkay = false;
}

//-------------------------------------------------------------------------

void wxScaledImageWindow::OnEraseBackground (wxEraseEvent & WXUNUSED(EraseEvent))
{
    // wxPanel::OnEraseBackground (EraseEvent);
}

void wxScaledImageWindow::OnPaint (wxPaintEvent & WXUNUSED(PaintEvent))
{
    wxPaintDC PaintWxDC (this);

    wxRect WindowRect = GetClientRect();
    bool IsOkay = false;

    if ((!m_IsCacheOkay)
        || (WindowRect != m_PreviousClientRect))
      {
        m_PreviousClientRect = WindowRect;
        wxImage ScaledImage;
        IsOkay = resizeImageVGrid (m_OriginalImage, 
                                   ScaledImage,
                                   WindowRect.GetWidth(), WindowRect.GetHeight());
        
        if (IsOkay)
          {
            m_ScaledBitmap = wxBitmap (ScaledImage);
            m_IsCacheOkay = true;
          }
      }
    else
      {
        IsOkay = true;
      }

    if (IsOkay)
      {
        PaintWxDC.DrawBitmap (m_ScaledBitmap, 0 /* x */, 0 /* y */, true /* transparent */);
      }
    else
      {
        PaintWxDC.SetPen (wxPen (*wxBLACK, 1, wxTRANSPARENT));
        PaintWxDC.SetBrush (wxBrush (wxColour (0xA0, 0xA0, 0xF0), wxSOLID));
        PaintWxDC.DrawRectangle (WindowRect);
      }
}

#   endif // wxUSE_GUI
#endif

//=========================================================================

#if wxUSE_GUI
IMPLEMENT_DYNAMIC_CLASS(wxItemWindow, wxWindow)

BEGIN_EVENT_TABLE (wxItemWindow, wxWindow)
    EVT_ERASE_BACKGROUND (wxItemWindow::OnEraseBackground)
    EVT_PAINT(wxItemWindow::OnPaint)
END_EVENT_TABLE()

//-------------------------------------------------------------------------

wxItemWindow::wxItemWindow ()
{
    m_IsSelectable = false;
    m_IsFocusable = false;
    m_StatusMessageTarget = NULL;

    init ();
}

wxItemWindow::wxItemWindow (wxWindow * ParentWindow, int ControlIdent,
                                    const wxPoint & Position, 
                                    const wxSize & Size,
                                    bool IsSelectable, bool IsFocusable,
                                    wxStatusMessageTarget * StatusMessageTarget,
                                    long Style, const wxString& Name)
  : wxWindow (ParentWindow, ControlIdent, 
              Position, Size,
              Style, Name)
{
    m_StatusMessageTarget = StatusMessageTarget;
    m_IsSelectable = IsSelectable;
    m_IsFocusable = IsFocusable;

    init ();
}

void wxItemWindow::init ()
{
    m_IsSelected = false;
    m_IsFocused = false;
    m_IsBorder = false;
    m_IsUsingFocusedBackgroundBrush = false;
}

//-------------------------------------------------------------------------

void wxItemWindow::setFocusedBackgroundBrush (const wxBrush & Brush)
{
    m_IsUsingFocusedBackgroundBrush = true;
    m_FocusedBackgroundBrush = Brush;
}

//-------------------------------------------------------------------------

void wxItemWindow::OnEraseBackground (wxEraseEvent & WXUNUSED(EraseEvent))
{
    // wxWindow::OnEraseBackground (EraseEvent);
}

void wxItemWindow::OnPaint (wxPaintEvent & WXUNUSED(PaintEvent))
{
    // wxRect WindowRect = GetRect ();
    // wxSize ClientSize = GetClientSize();

    wxPaintDC PaintWxDC (this);

    wxRect WindowRect = GetClientRect();

    // NOTE BorderWidth must be <= FocusBorderWidth (if m_IsFocusable)
    // or <= SelectionBorderWidth (if m_IsSelectable && !m_IsFocusable)
    enum { FocusBorderWidth = 3, SelectionBorderWidth = 3, BorderWidth = 1 };

    // calculate rectangles for borders and item
    wxRect ItemRect = WindowRect;
    wxRect SelectionIndicationRect = WindowRect;
    if (m_IsBorder)
      {
        if (!m_IsFocusable && !m_IsSelectable)
          {
            ItemRect.Inflate (-BorderWidth -1, -BorderWidth -1);
            SelectionIndicationRect.Inflate (-BorderWidth -1, -BorderWidth -1);
          }
      }

    if (m_IsFocusable)
      {
        ItemRect.Inflate (-FocusBorderWidth -1, -FocusBorderWidth -1);
        // ItemRect.Offset (FocusBorderWidth, FocusBorderWidth);
        SelectionIndicationRect.Inflate (-FocusBorderWidth -1, -FocusBorderWidth -1);
        // SelectionIndicationRect.Offset (FocusBorderWidth, FocusBorderWidth);
      }

    if (m_IsSelectable)
      {
        ItemRect.Inflate (-SelectionBorderWidth, -SelectionBorderWidth);
        // ItemRect.Offset (SelectionBorderWidth, SelectionBorderWidth);
      }

    wxBrush NormalBackgroundBrush = wxBrush (GetBackgroundColour(), wxSOLID);
    const wxBrush * BackgroundBrush = NULL;
    if (m_IsFocusable 
        && m_IsFocused
        && m_IsUsingFocusedBackgroundBrush)
      {
        BackgroundBrush = & m_FocusedBackgroundBrush;
      }
    else
      {
        BackgroundBrush = & NormalBackgroundBrush;
      }

    // draw background "behind" borders and rectangles (if there are any 
    // rectangles to be drawn)
    // NOTE this creates a minimal flicker if rectangles are drawn on the
    // rectangle drawn here
    if (WindowRect != ItemRect)
      {
        wxRegion BackgroundRegion (WindowRect);
        BackgroundRegion.Subtract (ItemRect);
        wxRegionIterator BackgroundRegionIterator (BackgroundRegion);
        PaintWxDC.SetBrush (* BackgroundBrush);
        PaintWxDC.SetPen (* wxTRANSPARENT_PEN);
        while (BackgroundRegionIterator.HaveRects ())
          {
            PaintWxDC.DrawRectangle (BackgroundRegionIterator.GetRect());
            ++BackgroundRegionIterator;
          }
      }
    
    // draw borders and rectangles around item
    if (m_IsBorder)
      {
        if ((!m_IsFocusable && !m_IsSelectable)
            || (!m_IsFocusable && m_IsSelectable && !m_IsSelected)
            || (m_IsFocusable && !m_IsFocused))
          {
            // draw gray rectangle to indicate area
            PaintWxDC.SetBrush (* wxTRANSPARENT_BRUSH);
            PaintWxDC.SetPen (wxPen (wxColour (0xD0, 0xD0, 0xD0), BorderWidth, wxSOLID));
            PaintWxDC.DrawRectangle (WindowRect.GetX(), 
                                     WindowRect.GetY(),
                                     WindowRect.GetWidth(), 
                                     WindowRect.GetHeight());
          }
      }

    if (m_IsFocusable)
      {
        if (m_IsFocused)
          {
            // draw blue rectangle around image to indicate selection:
            // and draw background with a very light gray
            // PaintWxDC.SetBrush (wxBrush (wxColour (0xF0, 0xF0, 0xF0), wxSOLID));
            if (!m_IsUsingFocusedBackgroundBrush)
              {
                PaintWxDC.SetBrush (* wxTRANSPARENT_BRUSH);
              }
            else
              {
                PaintWxDC.SetBrush (* wxTRANSPARENT_BRUSH);
                // PaintWxDC.SetBrush (m_FocusedBackgroundBrush);
              }
            PaintWxDC.SetPen (wxPen (wxColour (192, 32, 32), FocusBorderWidth, wxSOLID));
            PaintWxDC.DrawRectangle (WindowRect.GetX(), 
                                     WindowRect.GetY(),
                                     WindowRect.GetWidth(), 
                                     WindowRect.GetHeight());
          }
      }

    if (m_IsSelectable)
      {
        if (m_IsSelected)
          {
            // draw blue rectangle around image to indicate selection:
            PaintWxDC.SetBrush (* wxTRANSPARENT_BRUSH);
            PaintWxDC.SetPen (wxPen (wxColour (32, 0, 160), SelectionBorderWidth, wxSOLID));
            PaintWxDC.DrawRectangle (SelectionIndicationRect.GetX(), 
                                     SelectionIndicationRect.GetY(),
                                     SelectionIndicationRect.GetWidth(), 
                                     SelectionIndicationRect.GetHeight());
          }
      }

    // draw item
    handleItemPaint (PaintWxDC, ItemRect, BackgroundBrush);
}

void wxItemWindow::handleItemPaint (wxPaintDC & PaintWxDC,
                                    const wxRect & ItemRect,
                                    const wxBrush * BackgroundBrush)
{
    PaintWxDC.SetBrush (* BackgroundBrush);
    PaintWxDC.SetPen (* wxTRANSPARENT_PEN);
    PaintWxDC.DrawRectangle (ItemRect);
}
#endif // wxUSE_GUI

//=========================================================================

#if defined(__WXMSW__)

#if wxUSE_GUI
void drawDIBFitted (wxDC & WxDC, int X, int Y, int Width, int Height,
                    WXHDIB DIBHandle,
                    const wxBrush * BackgroundBrush)
{
    for_once
      {
        if (DIBHandle == NULL)
          break;

        // get DIB width and height:
        BITMAPINFO * BitmapInfo = (LPBITMAPINFO) GlobalLock (DIBHandle);
        if (BitmapInfo == NULL)
          break;

        BITMAPINFOHEADER * BitmapInfoHeader = (LPBITMAPINFOHEADER) BitmapInfo;

        int ImageWidth = BitmapInfoHeader -> biWidth;
        int ImageHeight = BitmapInfoHeader -> biHeight;
        int ColorPaletteSize = getActualUsedColorCount (BitmapInfo) * sizeof (RGBQUAD);

        unsigned char * BitmapData = ((unsigned char *) BitmapInfo)
          + BitmapInfoHeader -> biSize
          + ColorPaletteSize;

        wxRect Rect (X, Y, Width, Height);
        wxSize Size (ImageWidth, ImageHeight);
        wxRect FittedRect;
        fitInside (Rect, Size, FittedRect, wxALIGN_CENTER_HORIZONTAL | wxALIGN_CENTER_VERTICAL);

        FittedRect.Inflate (-1);

        // draw background around image if desired
        if (BackgroundBrush != NULL)
          {
            wxRegion BackgroundRegion (X, Y, Width, Height);
            BackgroundRegion.Subtract (FittedRect);
            wxRegionIterator BackgroundRegionIterator (BackgroundRegion);
            while (BackgroundRegionIterator.HaveRects ())
              {
                WxDC.SetBrush (* BackgroundBrush);
                WxDC.SetPen (* wxTRANSPARENT_PEN);
                WxDC.DrawRectangle (BackgroundRegionIterator.GetRect());
                ++BackgroundRegionIterator;
              }
          }
    
        // draw DIB
        HDC DCHandle = (HDC) WxDC.GetHDC();
        ::SetStretchBltMode (DCHandle, COLORONCOLOR);
        ::StretchDIBits (DCHandle,
                         FittedRect.GetX(), FittedRect.GetY(), 
                         FittedRect.GetWidth(), FittedRect.GetHeight(),
                         0, 0, ImageWidth, ImageHeight,
                         (void *) BitmapData,
                         BitmapInfo,
                         DIB_RGB_COLORS, SRCCOPY);

        FittedRect.Inflate (1);

        WxDC.SetBrush (*wxTRANSPARENT_BRUSH);
        WxDC.SetPen (*wxBLACK_PEN);
        WxDC.DrawRectangle (FittedRect.GetX(), FittedRect.GetY(),
                            FittedRect.GetWidth(), FittedRect.GetHeight());

        GlobalUnlock (DIBHandle);
    }
}
#endif // wxUSE_GUI

#endif

//=========================================================================

#if defined(__WXMSW__)

#if wxUSE_GUI
wxImageProxy::wxImageProxy ()
{
    m_DIBFromFile = NULL;
}

wxImageProxy::~wxImageProxy ()
{
    freeDIB ();
}

void wxImageProxy::freeDIB ()
{
    if (m_DIBFromFile != NULL)
      {
        GlobalFree (m_DIBFromFile);
        m_DIBFromFile = NULL;
      }
}

void wxImageProxy::discard ()
{
    freeDIB ();
}

void wxImageProxy::draw (wxDC & PaintWxDC, int X, int Y, int Width, int Height,
                         EResolutionIdent ResolutionIdent,
                         const wxBrush * BackgroundBrush)
{
    for_once
      {
        WXHDIB DIBHandle = NULL;
        getDIB (ResolutionIdent, DIBHandle);
        if (DIBHandle == NULL)
          break;

        drawDIBFitted (PaintWxDC, X, Y, Width, Height,
                       DIBHandle,
                       BackgroundBrush);
      }
}
#endif // wxUSE_GUI

#endif // defined(__WXMSW__)

//=========================================================================

#if defined(__WXMSW__)

#if wxUSE_GUI
wxBitmapPaintHelper::wxBitmapPaintHelper (const wxString & ResourceIdentString)
{
    loadBitmap (ResourceIdentString);
};

wxBitmapPaintHelper::wxBitmapPaintHelper (const wxString & ResourceIdentString, HDC PaintDCHandle)
{
    loadBitmap (ResourceIdentString);
    prepare (PaintDCHandle);
};

wxBitmapPaintHelper::wxBitmapPaintHelper (wxBitmap & Bitmap)
{
    setBitmap (Bitmap);
};

wxBitmapPaintHelper::wxBitmapPaintHelper (wxBitmap & Bitmap, HDC PaintDCHandle)
{
    setBitmap (Bitmap);
    prepare (PaintDCHandle);
};

wxBitmapPaintHelper::wxBitmapPaintHelper (int Width, int Height)
{
    createBitmap (Width, Height);
};

wxBitmapPaintHelper::wxBitmapPaintHelper (int Width, int Height, HDC PaintDCHandle)
{
    createBitmap (Width, Height);
    prepare (PaintDCHandle);
};

wxBitmapPaintHelper::~wxBitmapPaintHelper ()
{
    release ();
};

void wxBitmapPaintHelper::setBitmap (wxBitmap & Bitmap)
{
    m_BitmapRef = & Bitmap;
    m_BitmapHandle = (HBITMAP) m_BitmapRef -> GetHBITMAP ();
}

void wxBitmapPaintHelper::loadBitmap (const wxString & ResourceIdentString)
{
    // m_BitmapHandle = ::LoadBitmap (GetModuleHandle(NULL), MAKEINTRESOURCE(ResourceIdent));
    m_Bitmap.LoadFile (ResourceIdentString, wxBITMAP_TYPE_BMP_RESOURCE);
    m_BitmapRef = & m_Bitmap;
    m_BitmapHandle = (HBITMAP) m_BitmapRef -> GetHBITMAP ();
}

void wxBitmapPaintHelper::createBitmap (int Width, int Height)
{
    m_Bitmap.Create (Width, Height);
    m_BitmapRef = & m_Bitmap;
    m_BitmapHandle = (HBITMAP) m_BitmapRef -> GetHBITMAP ();
}

void wxBitmapPaintHelper::prepare (HDC PaintDCHandle)
{
    m_MemoryDCHandle = ::CreateCompatibleDC (PaintDCHandle);
    m_PreviousBitmapHandle = (HBITMAP) ::SelectObject (m_MemoryDCHandle, m_BitmapHandle);
}

void wxBitmapPaintHelper::release ()
{
    ::SelectObject (m_MemoryDCHandle, m_PreviousBitmapHandle);
    ::DeleteDC (m_MemoryDCHandle);
}

int wxBitmapPaintHelper::getWidth ()
{
    return m_BitmapRef -> GetWidth ();
}

int wxBitmapPaintHelper::getHeight ()
{
    return m_BitmapRef -> GetHeight ();
}
#endif // wxUSE_GUI

#endif // defined(__WXMSW__)

//=========================================================================

#if defined(__WXMSW__)

#if wxUSE_GUI
int getActualUsedColorCount (BITMAPINFO * BitmapInfo)
{
    int ColorsUsed = BitmapInfo -> bmiHeader.biClrUsed;
    // (for BI_RGB):
    // biClrUsed == 0 is interpreted differently depending on the
    // bit depth used:
    // - for color depths from 1..8 bpp: if biClrUsed is set to zero
    //   this means that a palette for all possible colors is there
    // - for color depths 16, 24 or 32 bpp: if biClrUsed is set to 
    //   zero this means that no palette is used (but biClrUsed
    //   may be set to non-zero if a palette is there for 
    //   output optimization purposes)
    if (ColorsUsed == 0 
        && BitmapInfo -> bmiHeader.biBitCount >= 1 
        && BitmapInfo -> bmiHeader.biBitCount <= 8) // FUZZY 8
      ColorsUsed = (1 << BitmapInfo -> bmiHeader.biBitCount);

    return ColorsUsed;
}
#endif // wxUSE_GUI

#endif // defined(__WXMSW__)

//=========================================================================

void WxBeep (int MilliSeconds)
{
    wxMinimalTimeInterval MinimalTimeInterval;
    MinimalTimeInterval.setMinimalTimeInterval (MilliSeconds);
    
    while (!MinimalTimeInterval.check ())
      {
        wxBell ();
      }
}

//=========================================================================

void doRotateMirrorCoord (int RotationAngle, 
                          bool IsHorizontallyMirrored, 
                          bool IsVerticallyMirrored,
                          const wxPoint & CentrePoint,
                          const wxPoint & InputPoint,
                          wxPoint & OutputPoint,
                          bool IsRotateCenter)
{
    wxPoint DistanceAsPoint = InputPoint - CentrePoint;

    wxPoint RotatedDistanceAsPoint = DistanceAsPoint;
    switch (RotationAngle)
      {
      case 90:
        RotatedDistanceAsPoint.x = DistanceAsPoint.y;
        RotatedDistanceAsPoint.y = -DistanceAsPoint.x;
        break;
      case 180:
        RotatedDistanceAsPoint.x = -DistanceAsPoint.x;
        RotatedDistanceAsPoint.y = -DistanceAsPoint.y;
        break;
      case 270:
        RotatedDistanceAsPoint.x = -DistanceAsPoint.y;
        RotatedDistanceAsPoint.y = DistanceAsPoint.x;
        break;
      }
    
    // PERFORMANCE NOTE doing the negation is possibly faster than 
    // multiplicating with (-1)
    wxPoint MirroredDistanceAsPoint = RotatedDistanceAsPoint;
    if (IsHorizontallyMirrored)
      {
        MirroredDistanceAsPoint.x = -RotatedDistanceAsPoint.x;
      }
    if (IsVerticallyMirrored)
      {
        MirroredDistanceAsPoint.y = -RotatedDistanceAsPoint.y;
      }

    wxPoint NewCentrePoint;
    if (IsRotateCenter
        && (RotationAngle == 90 || RotationAngle == 270))
      {
        NewCentrePoint.x = CentrePoint.y;
        NewCentrePoint.y = CentrePoint.x;
      }
    else
      {
        NewCentrePoint.x = CentrePoint.x;
        NewCentrePoint.y = CentrePoint.y;
      }
    OutputPoint = NewCentrePoint + MirroredDistanceAsPoint;
}

//=========================================================================

wxRotationMirrorState::wxRotationMirrorState()
{
    m_IsHorizontallyMirrored = false;
    m_IsVerticallyMirrored = false;
    m_RotationAngle = 0;

    m_IsUpdateRequired = true;
}

wxRotationMirrorState::~wxRotationMirrorState()
{
}

void wxRotationMirrorState::reset()
{
    m_IsHorizontallyMirrored = false;
    m_IsVerticallyMirrored = false;
    m_RotationAngle = 0;

    m_IsUpdateRequired = true;
}

//-------------------------------------------------------------------------

void wxRotationMirrorState::mirror (bool IsHorizontallyMirrored, bool IsVerticallyMirrored)
{
//  handleMirrorRotation (m_RotationAngle,
//                        IsHorizontallyMirrored,
//                        IsVerticallyMirrored);

    if (IsHorizontallyMirrored)
      {
        m_IsHorizontallyMirrored = !m_IsHorizontallyMirrored;
      }
    if (IsVerticallyMirrored)
      {
        m_IsVerticallyMirrored = !m_IsVerticallyMirrored;
      }

    m_IsUpdateRequired = true;
}

void wxRotationMirrorState::rotate (int RotationAngle)
{
    m_RotationAngle += RotationAngle;
    m_RotationAngle = (((int) m_RotationAngle) / 90) * 90;
    if (m_RotationAngle < 0)
      m_RotationAngle += 360;
    m_RotationAngle %= 360;

    handleMirrorRotation (RotationAngle,
                          m_IsHorizontallyMirrored,
                          m_IsVerticallyMirrored);

    m_IsUpdateRequired = true;
}

void wxRotationMirrorState::handleMirrorRotation(int RotationAngle,
                                                bool & IsHorizontallyMirrored,
                                                bool & IsVerticallyMirrored) 
{
    if (RotationAngle < 0)
      RotationAngle += 360;
    RotationAngle %= 360;
    if (RotationAngle == 90 
        || RotationAngle == 270)
      {
        if ((IsHorizontallyMirrored && !IsVerticallyMirrored)
            || (!IsHorizontallyMirrored && IsVerticallyMirrored))
          {
            // swap mirroring axes
            bool TempVerticallyMirrored = IsVerticallyMirrored;
            IsVerticallyMirrored = IsHorizontallyMirrored;
            IsHorizontallyMirrored = TempVerticallyMirrored;
          }
      }
}

void wxRotationMirrorState::getNormalizedState(int & RotationAngle,
                                              bool & IsHorizontallyMirrored,
                                              bool & IsVerticallyMirrored) 
{
    IsHorizontallyMirrored = m_IsHorizontallyMirrored;
    IsVerticallyMirrored = m_IsVerticallyMirrored;
    RotationAngle = m_RotationAngle;

    // normalize to reduce number of required image transformations:
    if (IsHorizontallyMirrored 
        && IsVerticallyMirrored)
      {
        RotationAngle += 180;
        RotationAngle %= 360;
        IsHorizontallyMirrored = false;
        IsVerticallyMirrored = false;
      }
}

//-------------------------------------------------------------------------

void wxRotationMirrorState::assign (const wxRotationMirrorState & RHSRotationMirrorState)
{
    m_RotationAngle = RHSRotationMirrorState.m_RotationAngle;
    m_IsHorizontallyMirrored = RHSRotationMirrorState.m_IsHorizontallyMirrored;
    m_IsVerticallyMirrored = RHSRotationMirrorState.m_IsVerticallyMirrored;

    m_IsUpdateRequired = true;
}

void wxRotationMirrorState::subtract (const wxRotationMirrorState & LHSRotationMirrorState,
                                      const wxRotationMirrorState & RHSRotationMirrorState)
{
    reset();

    int RotationAngleDifference = LHSRotationMirrorState.m_RotationAngle
      - RHSRotationMirrorState.m_RotationAngle;

    wxRotationMirrorState LHS;
    wxRotationMirrorState RHS;
    LHS.assign (LHSRotationMirrorState);
    RHS.assign (RHSRotationMirrorState);
    
    // rotate LHS to rotation angle of RHS, then determine differences in
    // mirroring, then apply mirroring and finally rotate
    LHS.rotate (- RotationAngleDifference);
    m_IsHorizontallyMirrored = RHS.m_IsHorizontallyMirrored != LHS.m_IsHorizontallyMirrored;
    m_IsVerticallyMirrored = RHS.m_IsVerticallyMirrored != LHS.m_IsVerticallyMirrored;
    rotate (RotationAngleDifference);
    
    m_IsUpdateRequired = true;
}

//=========================================================================

#ifdef M_WxExtLib_IsFilteredDIBEnabled

#if defined(__WXMSW__)

#if wxUSE_GUI
wxFilteredDIB::wxFilteredDIB ()
{
    m_Brightness = 0.;
    m_Contrast = 0.;
    m_RedGamma = 1.;
    m_GreenGamma = 1.;
    m_BlueGamma = 1.;
    m_IsInverted = false;
    m_IsColorAdjustmentModified = true;

    // m_IsPaletteBasedColorAdjustmentEnabled = true;
    m_IsOriginalColorPaletteRemembered = false;
    m_IsModifiedColorPaletteRemembered = false;

    m_IsImageMagickFilteringRequired = false;
    m_IsFilteringModified = true;

    m_PreviousRotationMirrorState.reset ();
    m_RotationMirrorState.reset();
    m_RotationMirrorState.m_IsUpdateRequired = true;

    m_SourceDIBHandle = NULL;
    m_IsOwningSourceImage = false;
    m_ResultDIBHandle = NULL;
    m_IsOwningResultImage = true;

    m_FilteredDIBBitsPerPixel = 24;
}

wxFilteredDIB::~wxFilteredDIB ()
{
    freeResultImage ();
    freeSourceImage ();
}

//-------------------------------------------------------------------------

void wxFilteredDIB::setBrightnessContrast (double Brightness, double Contrast)
{
    m_Brightness = Brightness;
    m_Contrast = Contrast;
    m_IsColorAdjustmentModified = true;
}

void wxFilteredDIB::setColorAdjustment (double RedGamma, double GreenGamma,
                                       double BlueGamma)
{
    m_RedGamma = RedGamma;
    m_GreenGamma = GreenGamma;
    m_BlueGamma = BlueGamma;
    m_IsColorAdjustmentModified = true;
}

void wxFilteredDIB::setInversion (bool IsInverted)
{
    m_IsInverted = IsInverted;
    m_IsColorAdjustmentModified = true;
}

//-------------------------------------------------------------------------

bool wxFilteredDIB::rememberPalette (WXHDIB DIBHandle,
                                       wxFilteredDIBColorPaletteArray & ColorPaletteByteArray)
{
    int DIBColorPaletteEntryCount = ::GetActualUsedColorCountInDIBHandle (DIBHandle);

    ColorPaletteByteArray.Clear ();

    if (DIBColorPaletteEntryCount == 0)
      return false;

    ColorPaletteByteArray.Insert ((BYTE) 0, /* array index */ 0,
                                  DIBColorPaletteEntryCount);

    // HACK WARN access of items in m_OriginalColorPaletteByteArray
    ::ImportExportColorPaletteIntoDIB ((BYTE *) (& ColorPaletteByteArray.Item(0)),
                                       DIBColorPaletteEntryCount,
                                       DIBHandle,
                                       false);

    return true;
}

bool wxFilteredDIB::restorePalette (WXHDIB DIBHandle,
                                      wxFilteredDIBColorPaletteArray & ColorPaletteByteArray)
{
    int ColorPaletteEntryCount = ColorPaletteByteArray.GetCount();
    
    if (ColorPaletteEntryCount != 0)
      {
        // HACK WARN access of items in m_OriginalColorPaletteByteArray
        ::ImportExportColorPaletteIntoDIB ((BYTE *) (& ColorPaletteByteArray.Item(0)),
                                           ColorPaletteEntryCount,
                                           DIBHandle,
                                           true);
      }

    return true;
}

void wxFilteredDIB::rememberOriginalPalette ()
{
    rememberPalette (m_SourceDIBHandle,
                     m_OriginalColorPaletteByteArray);
    m_IsOriginalColorPaletteRemembered = true;
}

void wxFilteredDIB::rememberModifiedPalette ()
{
    rememberPalette (m_SourceDIBHandle,
                     m_ModifiedColorPaletteByteArray);
    m_IsModifiedColorPaletteRemembered = true;
}

void wxFilteredDIB::setOriginalPaletteIntoSourceImage ()
{
    if (m_IsOriginalColorPaletteRemembered)
      {
        restorePalette (m_SourceDIBHandle,
                        m_OriginalColorPaletteByteArray);
      }
}

void wxFilteredDIB::setModifiedPaletteIntoSourceImage ()
{
    if (m_IsModifiedColorPaletteRemembered)
      {
        restorePalette (m_SourceDIBHandle,
                        m_ModifiedColorPaletteByteArray);
      }
}

bool wxFilteredDIB::getIsUsePaletteAlienation ()
{
    return false;
}

bool wxFilteredDIB::applyPaletteAlienation (wxFilteredDIBColorPaletteArray & ColorPaletteByteArray)
{
    return false;
}

//-------------------------------------------------------------------------

void wxFilteredDIB::resetRotationMirror ()
{
    m_RotationMirrorState.reset();
}

void wxFilteredDIB::rotate (int RotationAngle)
{
    m_RotationMirrorState.rotate (RotationAngle);
}

void wxFilteredDIB::mirror (bool IsHorizontallyMirrored, bool IsVerticallyMirrored)
{
    m_RotationMirrorState.mirror (IsHorizontallyMirrored, IsVerticallyMirrored);
}

//-------------------------------------------------------------------------

void wxFilteredDIB::setIsFilteringModified ()
{
    m_IsFilteringModified = true;
}

void wxFilteredDIB::setIsImageMagickFilteringRequired (bool IsRequired)
{
    m_IsImageMagickFilteringRequired = IsRequired;
    m_IsFilteringModified = true;
}

//-------------------------------------------------------------------------

void wxFilteredDIB::setSourceImage (const WXHDIB & SourceDIBHandle)
{
    m_SourceDIBHandle = SourceDIBHandle;
    m_IsColorAdjustmentModified = true;
    m_IsFilteringModified = true;
}

void wxFilteredDIB::getResultImage (WXHDIB & ResultDIBHandle)
{
    if (m_ResultDIBHandle == NULL)
      ResultDIBHandle = m_SourceDIBHandle;
    else
      ResultDIBHandle = m_ResultDIBHandle;
}

void wxFilteredDIB::setIsOwningSourceImage (bool IsOwningSourceImage)
{
    m_IsOwningSourceImage = IsOwningSourceImage;
}

void wxFilteredDIB::setIsOwningResultImage (bool IsOwningResultImage)
{
    m_IsOwningResultImage = IsOwningResultImage;
}

//-------------------------------------------------------------------------

void wxFilteredDIB::updateResultImage()
{
    if (! (m_IsColorAdjustmentModified
           || m_IsFilteringModified
           || m_RotationMirrorState.m_IsUpdateRequired))
      return;

    // rotation mirror optmization: avoid applying color adjustment and filtering
    // if only rotation or mirroring has changed - this assumes that rotation 
    // and mirroring doesn't change image contents (apart from rotation and 
    // mirroring)
    bool IsRotationMirrorOnly = m_RotationMirrorState.m_IsUpdateRequired
      && ! m_IsColorAdjustmentModified
      && ! m_IsFilteringModified;

    for_once
      {
        if (!IsRotationMirrorOnly)
          freeResultImage ();

        // obtain original DIB
        if (m_SourceDIBHandle == NULL)
            break;

        //-------------------------------------------------------------------------
        // try to apply color adjustment to palette DIBs

        bool IsPaletteBasedColorAdjustmentDone = false;

        if (m_IsPaletteBasedColorAdjustmentEnabled
            && (! IsRotationMirrorOnly))
          {
            // do palette modification using an ImageMagick filter - to this end, 
            // convert to 'image', then run filtering, then write result back
            // into the DIB palette
            ExceptionInfo exceptionInfo;
            GetExceptionInfo (& exceptionInfo);
    
            Image * PaletteMagickImage = NULL;

            if (!m_IsOriginalColorPaletteRemembered)
              {
                rememberOriginalPalette ();
              }
            else
              {
                setOriginalPaletteIntoSourceImage ();
              }

            // WARN DIBColorPaletteEntryCount may be only palette for optimization 
            // purposes, which would have to be ignored
            int DIBColorPaletteEntryCount = ::GetActualUsedColorCountInDIBHandle (m_SourceDIBHandle);
            bool IsPaletteImage = DIBColorPaletteEntryCount > 0;
            if (IsPaletteImage)
              {
                // use m_ModifiedColorPaletteByteArray as temporary buffer
                // to apply palette alienation and modification
                rememberPalette (m_SourceDIBHandle, m_ModifiedColorPaletteByteArray);

                if (IsPaletteImage
                    && getIsUsePaletteAlienation())
                  {
                    applyPaletteAlienation (m_ModifiedColorPaletteByteArray);
                  }

                const char * MapString = "BGRA"; // or maybe RGBO
                RGBQUAD * PaletteRGBAArray = (RGBQUAD *) & (m_ModifiedColorPaletteByteArray[0]);
                // transfer color values from palette to ImageMagick Image: 
                // the whole palette is put into a single row with DIBColorPaletteEntryCount
                // "pixels"
                PaletteMagickImage = ConstituteImage (DIBColorPaletteEntryCount, 
                                                      /* Height */ 1, 
                                                      MapString, 
                                                      CharPixel, 
                                                      PaletteRGBAArray,
                                                      & exceptionInfo);
                //
                //            bool IsPaletteImage = ::DIBPaletteToImage (m_SourceDIBHandle, 
                //                                                       & PaletteMagickImage,
                // & exceptionInfo);
                
                if (PaletteMagickImage == NULL)
                  break;

                // note that applyColorAdjustment() is an in-place operation
                applyColorAdjustment (& PaletteMagickImage, & exceptionInfo);
                if (PaletteMagickImage == NULL)
                  break;
            
                bool IsOk = ::ImageToDIBPalette (PaletteMagickImage, 
                                                 m_SourceDIBHandle,
                                                 & exceptionInfo);
                ::DestroyImage (PaletteMagickImage);
                PaletteMagickImage = NULL;

                // WARN should add exception/error handling/parsing after 
                // each function

                if (IsOk == false)
                  break;
                
                IsPaletteBasedColorAdjustmentDone = true;

                rememberModifiedPalette ();
              }
          }

        WXHDIB IntermediateDIBHandle = m_SourceDIBHandle; 
        WXHDIB PreviousIntermediateDIBHandle = m_SourceDIBHandle;
        
        //-------------------------------------------------------------------------
        // convert to ImageMagick image and apply color adjustment and ImageMagick filters

        if ((m_IsImageMagickFilteringRequired
             || !IsPaletteBasedColorAdjustmentDone)
            && (!IsRotationMirrorOnly))
          {
            // do ImageMagick filtering - to this end, convert to 'image'
            // then run filtering, then convert back to DIB
            ExceptionInfo exceptionInfo;
            GetExceptionInfo (& exceptionInfo);
    
            Image * MagickImage = ::DIBToImage (PreviousIntermediateDIBHandle, 
                                                & exceptionInfo);
            if (MagickImage == NULL)
              break;

            // check not to do color adjustment twice
            if (! IsPaletteBasedColorAdjustmentDone)
              {
                applyColorAdjustment (& MagickImage, & exceptionInfo);
              }
    
            applyImageMagickFiltering (& MagickImage, & exceptionInfo);
            if (MagickImage == NULL)
              break;
            
            IntermediateDIBHandle = ::ImageToDIB (MagickImage, 
                                                  m_FilteredDIBBitsPerPixel,
                                                  & exceptionInfo);
            ::DestroyImage (MagickImage);
            MagickImage = NULL;
            if (IntermediateDIBHandle == NULL)
              break;

            // WARN should add exception/error handling/parsing after 
            // each function
            // throwException (exceptionInfo);
            // if (image)
            //    throwException (image->exception);
          }

        // clean up of temporary DIB's
        if (IntermediateDIBHandle != PreviousIntermediateDIBHandle
            && PreviousIntermediateDIBHandle != m_SourceDIBHandle)
          {
            freeImage (PreviousIntermediateDIBHandle);
          }
        PreviousIntermediateDIBHandle = IntermediateDIBHandle;

        //-------------------------------------------------------------------------
        // rotate and mirror:

        int RotationAngle;
        bool IsHorizontallyMirrored;
        bool IsVerticallyMirrored;
        if (IsRotationMirrorOnly)
          {
            wxRotationMirrorState RelativeRotationMirrorState;

            RelativeRotationMirrorState.subtract (m_RotationMirrorState,
                                                  m_PreviousRotationMirrorState);

            RelativeRotationMirrorState.getNormalizedState (RotationAngle, 
                                                            IsHorizontallyMirrored, 
                                                            IsVerticallyMirrored);

            // setup DIB handle for input to rotation function
            if (m_ResultDIBHandle != NULL)
              {
                // if NULL, PreviousIntermediateDIBHandle should be set to
                // m_SourceDIBHandle
                PreviousIntermediateDIBHandle = m_ResultDIBHandle;
              }
          }
        else
          {
            m_RotationMirrorState.getNormalizedState (RotationAngle, 
                                                      IsHorizontallyMirrored, 
                                                      IsVerticallyMirrored);
          }

        if (RotationAngle != 0
            || IsHorizontallyMirrored
            || IsVerticallyMirrored)
          {
            bool IsNewRotatedDIBHandle = false;
            applyImageRotationMirroring (RotationAngle,
                                         IsHorizontallyMirrored,
                                         IsVerticallyMirrored,
                                         PreviousIntermediateDIBHandle,
                                         IntermediateDIBHandle,
                                         IsNewRotatedDIBHandle);

            // release temporary memory
            // if (IsNewFilteredDIBHandle)
            //  freeImage (FilteredDIBHandle);
            if (IntermediateDIBHandle != PreviousIntermediateDIBHandle
                && PreviousIntermediateDIBHandle != m_SourceDIBHandle)
              {
                freeImage (PreviousIntermediateDIBHandle);
              }
            PreviousIntermediateDIBHandle = IntermediateDIBHandle;
          }

        if (PreviousIntermediateDIBHandle != m_SourceDIBHandle)
          {
            m_ResultDIBHandle = PreviousIntermediateDIBHandle;
          }
        else
          {
            m_ResultDIBHandle = NULL;
          }

        m_IsColorAdjustmentModified = false;
        m_IsFilteringModified = false;
        m_RotationMirrorState.m_IsUpdateRequired = false;
        m_PreviousRotationMirrorState.assign (m_RotationMirrorState);

        handleResultImageChanged ();
      }
}

//-------------------------------------------------------------------------

void wxFilteredDIB::handleResultImageChanged ()
{
}

//-------------------------------------------------------------------------

void wxFilteredDIB::applyColorAdjustment (struct _Image ** MagickImagePtrPtr,
                                            struct _ExceptionInfo * exceptionInfo)
{
    for_once
      {
        Image * MagickImage = *MagickImagePtrPtr;
        if (MagickImage == NULL)
          break;

        if (m_IsInverted)
          NegateImage (MagickImage, 0);

        // WARN ContrastImage() seems to use only the sign of 
        // the 'sharpen' parameter
        ContrastImage (MagickImage, (m_Contrast * 100));

        // support for brightness currently done by modifying the gamma
        // of each channel
        double RedGamma = m_Brightness + m_RedGamma;
        double GreenGamma = m_Brightness + m_GreenGamma;
        double BlueGamma = m_Brightness + m_BlueGamma;

        levelGamma (RedGamma);
        levelGamma (GreenGamma);
        levelGamma (BlueGamma);

        GammaImageChannel (MagickImage, RedChannel, RedGamma);
        GammaImageChannel (MagickImage, GreenChannel, GreenGamma);
        GammaImageChannel (MagickImage, BlueChannel, BlueGamma);
      }
}

void wxFilteredDIB::levelGamma (double & Gamma)
{
    if (Gamma < 1.)
      {
        Gamma = 1. - (1. - Gamma) * 0.75;
      }
    else if (Gamma > 1.)
      {
        Gamma = 1. + (Gamma - 1.) * 1.75;
      }
    if (Gamma < 0.1)
      {
        Gamma = 0.1;
      }
}

void wxFilteredDIB::applyImageMagickFiltering (struct _Image ** IMImage,
                                                 struct _ExceptionInfo * exceptionInfo)
{
}

void wxFilteredDIB::applyImageRotationMirroring (int RotationAngle,
                                                   bool IsHorizontallyMirrored,
                                                   bool IsVerticallyMirrored,
                                                   WXHDIB & InputDIBHandle,
                                                   WXHDIB & OutputDIBHandle,
                                                   bool & IsNewOutputDIBHandle)
{
    WXHDIB IntermediateDIBHandle = InputDIBHandle; 
    WXHDIB PreviousIntermediateDIBHandle = InputDIBHandle;

    // rotate image data if required
    switch (RotationAngle)
      {
      case 90:
        IntermediateDIBHandle = applyDIBRotation (PreviousIntermediateDIBHandle, 90);
        break;
      case 180:
        IntermediateDIBHandle = applyDIBRotation (PreviousIntermediateDIBHandle, 180);
        break;
      case 270:
        IntermediateDIBHandle = applyDIBRotation (PreviousIntermediateDIBHandle, 270);
        break;
      }

    // clean up of temporary DIB's
    if (IntermediateDIBHandle != PreviousIntermediateDIBHandle
        && PreviousIntermediateDIBHandle != InputDIBHandle)
      {
        freeImage (PreviousIntermediateDIBHandle);
      }
    PreviousIntermediateDIBHandle = IntermediateDIBHandle;

    // apply horizontal mirroring if required
    if (IsHorizontallyMirrored)
      {
        IntermediateDIBHandle = applyDIBMirroring (PreviousIntermediateDIBHandle, false);
      }

    // clean up of temporary DIB's
    if (IntermediateDIBHandle != PreviousIntermediateDIBHandle
        && PreviousIntermediateDIBHandle != InputDIBHandle)
      {
        freeImage (PreviousIntermediateDIBHandle);
      }
    PreviousIntermediateDIBHandle = IntermediateDIBHandle;

    // apply vertical mirroring if required
    if (IsVerticallyMirrored)
      {
        IntermediateDIBHandle = applyDIBMirroring (PreviousIntermediateDIBHandle, true);
      }

    // clean up of temporary DIB's
    if (IntermediateDIBHandle != PreviousIntermediateDIBHandle
        && PreviousIntermediateDIBHandle != InputDIBHandle)
      {
        freeImage (PreviousIntermediateDIBHandle);
      }
    PreviousIntermediateDIBHandle = IntermediateDIBHandle;

    // set output variables
    OutputDIBHandle = PreviousIntermediateDIBHandle;
    IsNewOutputDIBHandle = InputDIBHandle != OutputDIBHandle;
}

//-------------------------------------------------------------------------

void wxFilteredDIB::freeSourceImage()
{
    if (m_IsOwningSourceImage)
      {
        freeImage (m_SourceDIBHandle);
      }
}

void wxFilteredDIB::freeResultImage()
{
    if (m_IsOwningResultImage)
      {
        freeImage (m_ResultDIBHandle);
      }
}

void wxFilteredDIB::freeImage (WXHDIB & DIBHandle)
{
    if (DIBHandle != NULL)
      {
        GlobalFree (DIBHandle);
        DIBHandle = NULL;
      }
}
#endif // wxUSE_GUI

#endif // defined(__WXMSW__)

#endif

//=========================================================================

wxChoiceManager::wxChoiceManager ()
{
    m_FocusedSourceItemIndex = -1;
    m_FocusedChoiceItemIndex = -1;
}

void wxChoiceManager::OnSelect ()
{
    selectIntoChoice (getChoiceFocus(), getSourceFocus());
}

void wxChoiceManager::OnDeselect ()
{
    deselectFromChoice (getChoiceFocus());
}

void wxChoiceManager::OnClearSelection ()
{
    int ChoiceItemCount = getChoiceItemCount ();
    for (int ChoiceItemIndex=0; ChoiceItemIndex<ChoiceItemCount; ++ChoiceItemIndex)
      {
        deselectFromChoice (ChoiceItemIndex, false);
      }
    updateChoiceView ();
}

void wxChoiceManager::OnSourceLeftMouseButton (int SourceItemIndex)
{
    bool IsNewlyFocused = getSourceFocus () != SourceItemIndex;
        
    if (IsNewlyFocused)
      setSourceFocus (SourceItemIndex);

    handleSelection (IsNewlyFocused, getChoiceFocus (), SourceItemIndex);
}

void wxChoiceManager::OnSourceRightMouseButton (int SourceItemIndex)
{
    int DocumentIdent = getSourceDocument (SourceItemIndex);
    deselectFromChoice (findChoiceItemIndex (DocumentIdent));
}

void wxChoiceManager::OnChoiceLeftMouseButton (int ChoiceItemIndex)
{
    bool IsNewlyFocused = getChoiceFocus () != ChoiceItemIndex;

    if (IsNewlyFocused)
      setChoiceFocus (ChoiceItemIndex);

    handleSelection (IsNewlyFocused, ChoiceItemIndex, getSourceFocus());
}

void wxChoiceManager::OnChoiceRightMouseButton (int ChoiceItemIndex)
{
    deselectFromChoice (ChoiceItemIndex);
}

//-------------------------------------------------------------------------

void wxChoiceManager::handleSelection (bool IsNewlyFocused, int ChoiceItemIndex,
                                         int SourceItemIndex)
{
    bool IsChoiceHavingDocument =  getChoiceDocument (ChoiceItemIndex) != -1;
    bool IsDocumentAlreadyAssigned = findChoiceItemIndex (getSourceDocument (SourceItemIndex)) != -1;
    
    if ((!IsNewlyFocused || (!IsChoiceHavingDocument && !IsDocumentAlreadyAssigned))
        && SourceItemIndex != -1 
        && ChoiceItemIndex != -1)
      {
        selectIntoChoice (ChoiceItemIndex, SourceItemIndex);
      }
}

//-------------------------------------------------------------------------

void wxChoiceManager::selectIntoChoice (int ChoiceItemIndex, int SourceItemIndex,
                                                 bool IsUpdateChoiceView)
{
    int ChoiceItemCount = getChoiceItemCount();

    if (ChoiceItemIndex != -1 && SourceItemIndex != -1)
      {
        // create entries of -1 if necessary
        if (ChoiceItemIndex >= ChoiceItemCount)
          {
            insertChoiceDocument (-1, ChoiceItemCount, ChoiceItemIndex + 1 - ChoiceItemCount);
          }

        deselectFromChoice (ChoiceItemIndex, 
                             /* IsUpdateChoiceView */ false);

        setSelection (SourceItemIndex, true);
        int DocumentIdent = getSourceDocument (SourceItemIndex);
        setChoiceDocument (ChoiceItemIndex, DocumentIdent);

        if (IsUpdateChoiceView)
          updateChoiceView ();
      }
}

void wxChoiceManager::deselectFromChoice (int ChoiceItemIndex, 
                                                   bool IsUpdateChoiceView)
{
    if (ChoiceItemIndex != -1 && ChoiceItemIndex < getChoiceItemCount())
      {
        int DocumentIdent = getChoiceDocument (ChoiceItemIndex);

        if (DocumentIdent != -1)
          {
            int SourceItemIndex = findSourceItemIndex (DocumentIdent);
            setChoiceDocument (ChoiceItemIndex, -1);
            int OtherChoiceItemIndex = findChoiceItemIndex (DocumentIdent);
            setSelection (SourceItemIndex, OtherChoiceItemIndex != -1);

            if (IsUpdateChoiceView)
              updateChoiceView ();
          }
      }
}

//-------------------------------------------------------------------------

int wxChoiceManager::getSourceFocus ()
{
    return m_FocusedSourceItemIndex;
}

void wxChoiceManager::setSourceFocus (int NewFocusedItemIndex)
{
    if (NewFocusedItemIndex == m_FocusedSourceItemIndex)
      return;

    if (m_FocusedSourceItemIndex != -1)
      {
        updateSourceFocus (m_FocusedSourceItemIndex, false);
      }
    m_FocusedSourceItemIndex = NewFocusedItemIndex;

    if (m_FocusedSourceItemIndex != -1)
      {
        updateSourceFocus (m_FocusedSourceItemIndex, true);
      }
}

int wxChoiceManager::getChoiceFocus ()
{
    return m_FocusedChoiceItemIndex;
}

void wxChoiceManager::setChoiceFocus (int NewFocusedItemIndex)
{
    if (NewFocusedItemIndex == m_FocusedChoiceItemIndex)
      return;

    if (m_FocusedChoiceItemIndex != -1)
      {
        updateChoiceFocus (m_FocusedChoiceItemIndex, false);
      }
    m_FocusedChoiceItemIndex = NewFocusedItemIndex;

    if (m_FocusedChoiceItemIndex != -1)
      {
        updateChoiceFocus (m_FocusedChoiceItemIndex, true);
      }
}

void wxChoiceManager::redoSelections ()
{
    int SourceItemCount = getSourceItemCount ();
    for (int SourceItemIndex=0; SourceItemIndex<SourceItemCount; ++SourceItemIndex)
      {
        int DocumentIdent = getSourceDocument (SourceItemIndex);
        setSelection (SourceItemIndex, findChoiceItemIndex (DocumentIdent) != -1);
      }
    updateChoiceView ();
}

void wxChoiceManager::setSelection (int SourceItemIndex, bool IsSelect)
{
    if (SourceItemIndex == -1)
      return;

    updateSelection (SourceItemIndex, IsSelect);
}

//-------------------------------------------------------------------------

int wxChoiceManager::findSourceItemIndex (int DocumentIdent)
{
    int SourceDocumentCount = getSourceItemCount();

    for (int SourceDocumentIndex=0; SourceDocumentIndex<SourceDocumentCount; ++SourceDocumentIndex)
      {
        if (getSourceDocument (SourceDocumentIndex) == DocumentIdent)
          return SourceDocumentIndex;
      }

    return -1;
}

int wxChoiceManager::findChoiceItemIndex (int DocumentIdent)
{
    int ChoiceDocumentCount = getChoiceItemCount ();

    for (int ChoiceDocumentIndex=0; ChoiceDocumentIndex<ChoiceDocumentCount; ++ChoiceDocumentIndex)
      {
        if (getChoiceDocument (ChoiceDocumentIndex) == DocumentIdent)
          return ChoiceDocumentIndex;
      }

    return -1;
}

//=========================================================================

wxDefaultParentWindowFunc s_GlobalDefaultParentWindowFunc = NULL;

void wxSetGlobalDefaultParentWindowFunc (wxDefaultParentWindowFunc Func)
{
    s_GlobalDefaultParentWindowFunc = Func;
}

wxDefaultParentWindowFunc wxGetGlobalDefaultParentWindowFunc ()
{
    return s_GlobalDefaultParentWindowFunc;
}

wxWindow * wxGetDefaultParentWindow (wxWindow * ParentWindow)
{
    if (ParentWindow != NULL)
      return ParentWindow;

    if (s_GlobalDefaultParentWindowFunc != NULL)
      {
        return s_GlobalDefaultParentWindowFunc (ParentWindow);
      }

    // return ParentWindow, which is NULL here
    return ParentWindow;
}

//=========================================================================
