// -*- c++ -*-
/*
-------------------------------------------------------------------------
This file is part of WxWidgetsExtensions library.
-------------------------------------------------------------------------

WxExtLib (WxWidgetsExtensions) library
-----------------------------

COPYRIGHT NOTICE:

WxExtLib library Copyright (c) 2003-2007 Daniel Kps

The WxWidgetsExtensions library and associated documentation files (the
"Software") is provided "AS IS".  The author(s) disclaim all
warranties, expressed or implied, including, without limitation, the
warranties of merchantability and of fitness for any purpose.  The
author(s) assume no liability for direct, indirect, incidental,
special, exemplary, or consequential damages, which may result from
the use of or other dealings in the Software, even if advised of the
possibility of such damage.

Permission is hereby granted, free of charge, to any person obtaining
a copy of this Software, to deal in the Software without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Software,
and to permit persons to whom the Software is furnished to do so,
subject to the following conditions:

 1. The origin of this source code must not be misrepresented.
 2. Altered versions must be plainly marked as such and must not be
    misrepresented as being the original source.
 3. This Copyright notice may not be removed or altered from any 
    source or altered source distribution.

End of WxExtLib library Copyright Notice

-------------------------------------------------------------------------
*/

#ifndef _INCLUDED_WxExtDialog_h  
#define _INCLUDED_WxExtDialog_h  

#if defined(__GNUG__) && (!defined(__APPLE__)) && (!defined(M_NoPragmaInterface))
#   pragma interface "WxExtDialog.h"
#endif

#include "WxExtLibConfig.h"

#include <wx/defs.h>

#ifndef WX_PRECOMP
#   include <wx/dialog.h>
#   include <wx/intl.h>
#   include <wx/checkbox.h>
#   include <wx/choice.h>
#   include <wx/radiobut.h>
#   include <wx/statbmp.h>
#   include <wx/sizer.h>
#   include <wx/button.h>
#   include <wx/stattext.h>
#   include <wx/statbox.h>
#   include <wx/slider.h>
#   include <wx/listbox.h>
#   include <wx/combobox.h>
#   include <wx/valgen.h>
#endif

// for wxAppRootFrame only:
#include <wx/frame.h>

#include <wx/utils.h>
#include <wx/toplevel.h>
#include <wx/busyinfo.h>

#include "StatusValidators.h"
#include "WxMisc.h"

#include <accessor_macros.h>

//-------------------------------------------------------------------------

#define M_IsDefineExportMagic 1
#    include "WxExtLibAliases.h"
#undef M_IsDefineExportMagic

//-------------------------------------------------------------------------

// repeat define from 'wx/toplevel.h' for older wxWidgets versions (e.g. wx 2.4):
#define wxCLOSE_BOX 0x1000

// define for automatically assigned control idents (to improve readability
// of code somewhat)
#define wxID_AUTO (-1)


//-------------------------------------------------------------------------

#ifdef M_WxExtLib_IsMFCCompatibility
//-------------------------------------------------------------------------
// create a wxWindow as child of the specified MFC ParentWindow - this
// is useful for using wxWindows based dialogs together with a MFC
// based parent/frame window
extern wxWindow * MSWCreateChildWindowForMFCFrame (CWnd * ParentWindow);

#endif // M_WxExtLib_IsMFCCompatibility

//-------------------------------------------------------------------------

class wxSizerAdder;
extern wxSizerAdder & wxNullSizerAdder;

class wxSizerAdderProxy;
extern wxSizerAdderProxy & wxNullSizerAdderProxy;

//-------------------------------------------------------------------------
// wxExtDlgBase: helper class with following purposes:
// - ease creating and adding controls
// - slightly specialized dialog to work with a MFC main window 
// - support for dialog zoom
// - simplification of control-change notification
class wxExtDlgBase
{
public:

    wxExtDlgBase ();
    void initCWxDialog ();
    virtual ~wxExtDlgBase ();

    // getDialog() actually returns wxWindow, not wxDialog, and is therefore
    // (partially) compatible with e.g. wxFrame
    virtual wxWindow * getDialog () = 0;
  
    void setDialogInitializedFlag ();

#ifdef M_IsNonWxMainIntegration
    //-------------------------------------------------------------------------
    // find the window to use as parent for this dialog if none has been
    // specified explicitly by the user (requires patched version of
    // src/msw/dialog.h with this function made virtual)
    //
    // may return NULL
    virtual wxWindow *FindSuitableParent() const;
#endif

    // show modal dialog and enter modal loop
    // virtual void DoShowModal();

#ifdef M_IsNonWxMainIntegration
#if defined(__WXMSW__)
    // added for MFC main window compatibility:
    // MSWEnableOtherWindows (called by DoShowModal()). Can 
    // be overwritten for the case that the frame window 
    // was not created with wxWindows (requires a patched
    // version of src/msw/dialog.cpp)
    virtual void MSWEnableOtherWindows (bool IsEnable);
#endif
#endif
#if defined(__WXMSW__)
    HWND MSWGetHandle();
#endif


    //-------------------------------------------------------------------------
    // helper functions to make creation of certain often-used dialog control
    // combinations easier (e.g. a wxStaticText and wxTextCtrl may be created
    // with a single call using addTextCtrlWithLabel())
    // - for notes on size specifications with regard to zooming, see
    //   the getAdjusted*Width/Height functions

    wxSizer * addSpace (int Width, int Height, const wxSizerAdderProxy & SizerAdderProxy);

    wxWindow * getAddParentWindow (wxSizerAdder * SizerAdder);

    // DEF ButtonPtrPtr can be NULL
    // DEF ButtonId can be -1 to obtain an automatically generated button id
    wxButton * addButton (wxButton ** ButtonPtrPtr, wxWindowID id,
                          wxCommandEventFunction CommandEventFunction, 
                          const wxString& label, 
                          const wxSizerAdderProxy & SizerAdderProxy,
                          const wxPoint& pos = wxDefaultPosition,
                          const wxSize& size = wxDefaultSize,
                          long style = 0, const wxValidator& validator = wxDefaultValidator);

    wxSizer * addCheckBoxWithLabel (wxCheckBox ** CheckBox,
                                   const wxString & LabelString,
                                   const wxValidator & Validator = wxDefaultValidator,
                                   long Flags = 0,  // Flags can be 0 or wxTE_READONLY
                                   const wxSizerAdderProxy & SizerAdderProxy = wxNullSizerAdderProxy,
                                   bool IsLabelOnTheLeft = false);

    wxSizer * addRadioButtonWithLabel (wxRadioButton ** RadioButton,
                                    const wxString & LabelString,
                                    const wxValidator & Validator = wxDefaultValidator,
                                    long RadioCtrlStyle = 0, 
                                    const wxSizerAdderProxy & SizerAdderProxy = wxNullSizerAdderProxy);

    wxStaticText * addLabel (wxStaticText ** StaticTextCtrl,
                             const wxString & LabelString,
                             const wxSizerAdderProxy & SizerAdderProxy = wxNullSizerAdderProxy,
                             long WindowStyle = wxALIGN_LEFT,
                             const wxSize & Size = wxDefaultSize,
                             bool IsTransparent = false,
                             long TransparentStaticTextFlags = 0);

    wxWindow * addMultiLineText (wxMultiLineText * MultiLineText,
                                 long CtrlStyle = 0,
                                 const wxSize & CtrlSize = wxDefaultSize,
                                 int MultiLineTextStyle = 0,
                                 const wxSizerAdderProxy & SizerAdderProxy = wxNullSizerAdderProxy);
    wxWindow * addMultiLineText (wxMultiLineText * MultiLineText,
                                 long CtrlStyle,
                                 const wxSize & CtrlSize,
                                 int MultiLineTextStyle,
                                 const wxSizerAdderProxy & SizerAdderProxy,
                                 const wxString & TextString,
                                 int MinLineCount, int MaxLineCount,
								 int BorderWidth=0);

    wxTextCtrl * addTextCtrl (wxTextCtrl ** TextCtrl,
                              const wxValidator & Validator = wxDefaultValidator,
                              long TextCtrlStyle = wxTE_LEFT,
                              const wxSize & Size = wxDefaultSize,
                              const wxSizerAdderProxy & SizerAdderProxy = wxNullSizerAdderProxy);

    // TODO add sizer flags for textctrl, e.g. Expandable TextCtrl, ExtraSizer,
    wxSizer * addTextCtrlWithLabel (wxTextCtrl ** TextCtrl,
                                       const wxString & LabelString,
                                       const wxValidator & Validator = wxDefaultValidator,
                                       long TextCtrlStyle = wxTE_LEFT, 
                                       const wxSize & Size = wxDefaultSize,
                                       const wxSizerAdderProxy & SizerAdderProxy = wxNullSizerAdderProxy);

    wxSizer * addTextCtrlWithTwoLabels (wxTextCtrl ** TextCtrl,
                                        const wxString & FirstLabelString,
                                        const wxString & SecondLabelString,
                                        const wxValidator & Validator = wxDefaultValidator,
                                        long TextCtrlStyle = wxTE_LEFT, 
                                        const wxSize & Size = wxDefaultSize,
                                        const wxSizerAdderProxy & SizerAdderProxy = wxNullSizerAdderProxy);

    // addTextCtrlSliderPairWithLabel():
    // - currently, only a horizontal wxSlider control can be created
    // TODO:
    // - add parameters for wxTextCtrl, wxSlider styles, add size parameters
    wxSizer * addTextCtrlSliderPairWithLabel (wxTextCtrl ** TextCtrl,
                                              wxSlider ** SliderCtrl,
                                              const wxString & LabelString,
                                              long * LongValue, long MinLong, long MaxLong,
                                              wxStatusMessageTarget * StatusMessageTarget,
                                              const wxSizerAdderProxy & SizerAdderProxy, 
                                              const wxString & FieldNameString);

    wxListBox * addListBox (wxListBox ** ListBox, 
                            const wxValidator & Validator,
                            int ChoiceCount, const wxString ChoiceStringArray [],
                            const wxSize& Size = wxDefaultSize,
                            const wxSizerAdderProxy & SizerAdderProxy = wxNullSizerAdderProxy);

    wxChoice * addChoiceCtrl (wxChoice ** ChoiceCtrl, 
                              const wxValidator & Validator,
                              int ChoiceCount, const wxString ChoiceStringArray [],
                              const wxSize& Size = wxDefaultSize,
                              const wxSizerAdderProxy & SizerAdderProxy = wxNullSizerAdderProxy);
    wxSizer * addChoiceCtrlWithLabel (wxChoice ** ChoiceCtrl,
                                      const wxString & LabelString,
                                      const wxValidator & Validator,
                                      int ChoiceCount, const wxString ChoiceStringArray [],
                                      const wxSize & Size = wxDefaultSize,
                                      const wxSizerAdderProxy & SizerAdderProxy = wxNullSizerAdderProxy);

    wxComboBox * addComboBox (wxComboBox ** ComboBox, 
                              const wxValidator & Validator,
                              int StringCount, const wxString StringArray [],
                              const wxSize& Size = wxDefaultSize,
                              const wxSizerAdderProxy & SizerAdderProxy = wxNullSizerAdderProxy);
    wxSizer * addComboBoxWithLabel (wxComboBox ** ComboBox,
                                    const wxString & LabelString,
                                    const wxValidator & Validator,
                                    int StringCount, const wxString StringArray [],
                                    const wxSize & Size = wxDefaultSize,
                                    const wxSizerAdderProxy & SizerAdderProxy = wxNullSizerAdderProxy);

    wxSizer * addStaticBitmap (wxStaticBitmap ** StaticBitmapPtr,
                               const unsigned char * ImageData,
                               size_t ImageDataSize,
                               const wxPoint& Position = wxDefaultPosition,
                               const wxSize& Size = wxDefaultSize,
                               const wxSizerAdderProxy & SizerAdderProxy = wxNullSizerAdderProxy,
                               long WindowStyle = 0);
    wxSizer * addStaticBitmap (wxStaticBitmap ** StaticBitmapPtr,
                               const char * ResourceNameString,
                               const wxPoint& Position,
                               const wxSize& Size,
                               const wxSizerAdderProxy & SizerAdderProxy,
                               long WindowStyle = 0);
    wxSizer * addStaticBitmap (wxStaticBitmap ** StaticBitmapPtr,
                               wxBitmap * Bitmap,
                               const wxPoint& Position,
                               const wxSize& Size,
                               const wxSizerAdderProxy & SizerAdderProxy,
                               long WindowStyle = 0);

    //-------------------------------------------------------------------------
    // creation of status message TextCtrl with required flags (e.g. wxTE_READONLY)

    wxMultiLineText * createErrorMessageCtrl (wxMultiLineText * ErrorMessageMultiLineText,
                                              int LineCount = 2);

    //-------------------------------------------------------------------------
    // TODO could write some Sizer for automatic horizontal or vertical layout

    void addFlexGridSizer (wxFlexGridSizer ** FlexGridSizer,
                           int Columns, int VGap, int HGap);
    void addFlexGridSizer (wxFlexGridSizer ** FlexGridSizer,
                           int Columns, int VGap, int HGap,
                           wxSizer * OuterSizer, int AddFlags, int AddBorder);

    void addStaticBox (wxFlexGridSizer ** FlexGridSizer,
                       const wxString & LabelString,
                       int Columns, int VGap, int HGap,
                       wxSizer * OuterSizer, int AddFlags, int AddBorder,
                       const wxSizerAdderProxy & SizerAdderProxy);

    //-------------------------------------------------------------------------

    // createDialogFont()
    // - it contains some 'handicrafted' logic to select a well-reable
    //   (not to small) font depending on the used platform (must specify
    //   wxDEFAULT as family for this)
    wxFont createDialogFont (int PointSize, 
                             int Family, 
                             int Style, 
                             int Weight, 
                             const bool IsUnderlined = false, 
                             const wxString& FaceName = "", 
                             wxFontEncoding Encoding = wxFONTENCODING_DEFAULT,
                             int * RequestedPointSize = NULL,
                             bool IsCreateFont = true);

    //-------------------------------------------------------------------------
    // functions for calculating pixel widths/heights based on some desired
    // zoom value
    // Notes:
    // - The getAdjusted...Size() functions in part use 'handicrafted' formulas (e.g.
    //   getAdjustedTextCtrlHeight(), getAdjustedButtonHeight()), specific
    //   to each of the different platforms/window systems (currently, 
    //   available for wxMSW, wxX11, wxGTK, wxMotif). 
    // - Currently, these functions assume a font size equal to the font
    //   size set for the dialog (e.g. set via setDialogZoom()).
    // - Currently, the function which creates controls must call the
    //   getAdjusted*Width/Height functions itself. For cases where this
    //   flexibility is not needed, a mode should be added so that 
    //   addTextCtrl()/addSlider etc. functions call the corresponding
    //   getAdjusted*Width/Height() functions implicitely.
    virtual int getAdjustedBorderSize (int BorderSize);
    virtual int getAdjustedTextCtrlWidth (int TextCtrlWidth);
    virtual int getAdjustedTextCtrlHeight (int TextCtrlHeight, bool IsFromLineCount = false);
    virtual int getAdjustedChoiceCtrlWidth (int ChoiceCtrlWidth);
    virtual int getAdjustedChoiceCtrlHeight (int ChoiceCtrlHeight, bool IsFromLineCount = false);
    virtual int getAdjustedListBoxWidth (int ListBoxWidth);
    virtual int getAdjustedListBoxHeight (int ListBoxHeight, bool IsFromLineCount = false);
    virtual int getAdjustedComboBoxWidth (int ComboBoxWidth);
    virtual int getAdjustedComboBoxHeight (int ComboBoxHeight, bool IsFromLineCount = false);
    virtual int getAdjustedButtonWidth (int ButtonWidth);
    virtual int getAdjustedButtonHeight ();
    virtual int getAdjustedSliderWidth (int SliderWidth, bool IsVertical = false);
    virtual int getAdjustedSliderHeight (int SliderHeight, bool IsVertical = false);
    virtual int getAdjustedBitmapWidth (int BitmapWidth);
    virtual int getAdjustedBitmapHeight (int BitmapHeight);

    virtual void adjustMultiLineTextSize (wxMultiLineText & MultiLineText,
                                          int MinLineCount, int MaxLineCount);

    int getDefaultDialogFontSize ();
    int getDialogFontSize ();
    int getDialogFontHeight ();

    void setIsSetFontIfSameAsDefaultSize (bool IsSetFontIfSameAsDefaultSize);

    virtual int getAdjustedProportionalWidth(int Width);
    virtual int getAdjustedProportionalHeight(int Height);
    virtual int getAdjustedFontLikeWidth(int Width);
    virtual int getAdjustedFontLikeHeight(int Height);

    // aliases for less typing overhead:
    int _B(int BorderSize) { return getAdjustedBorderSize (BorderSize); } 

    void setDefaultButtonSize (int Width, int Height);

    void setHtmlWindowFonts (wxHtmlWindow * HtmlWindow,
                             int BaseSize,
                             bool IsUseReadableFontSizes = true,
                             bool IsUseScaleableFont = true);
    void excludeRange (int & PointSize, int ExcludeRangeMin, int ExcludeRangeMax = -1);

    //-------------------------------------------------------------------------
    //-- dialog zoom support:

    // setDialogZoom() should be called before getAdjustedBorderSize() etc.
    // is used, because calling setDialogZoom() after having created sizers
    // with absolute pixel widths/heights/border sizes etc., setDialogZoom()
    // has no effect
    // ZoomValue should be something between 0.5 and 1.5
    // a value of 1. means no zooming
    virtual double getZoom ();
    void setDialogZoom (double DialogZoomValue);
    double getDialogZoom ();
    static void setGlobalDialogZoom (double ZoomValue);
    static double getGlobalDialogZoom ();
    void setDesktopReferenceSize (const wxSize & Size);
    static void setGlobalDesktopReferenceSize (const wxSize & Size);

    // BorderCompressionValue should be something between 0.5 and 1.0
    // a value of 1. means no border compression
    // border compression may be useful for non-modal dialogs which 
    // should occupy as least as possible space on the screen (while using
    // a still readable font) - using this function, the same dialog
    // can be used modal (with setting border compression to the default,
    // 1.) and non-modal (with setting border compression to e.g. 0.5)
    void setBorderCompression (double BorderCompressionValue);

    //-------------------------------------------------------------------------
    wxSize getSizeFromDefaults (const wxSize & Size, const wxSize & DefaultSize);
    wxSize getSizeFromAspectRatio (const wxSize & Size, const wxSize & AspectRatioAsSize);

    //-------------------------------------------------------------------------
    // support for control change notification
    // - most normal notifications are dispatched to handleWatchedControlChange()
    //   (call addWatchedControl() for desired controls)
    // - no need for static message handlers, useful if controls use 
    //   dynamically selected Id's

    virtual void enableWatchedControlChangeNotification (bool IsWatchedControlChangeNotification);
    virtual void addWatchedControl (wxWindow * Window);
    virtual void handleControlChange(wxCommandEvent & Event);
    virtual void handleWatchedControlChange(wxCommandEvent & Event);

    //-------------------------------------------------------------------------
    // dialog/window/system close event handling

    virtual void handleCloseEvent(wxCloseEvent & WXUNUSED(Event)) { }
    void OnCloseEvent (wxCloseEvent & CloseEvent);

    //-------------------------------------------------------------------------
    // notification functions called by the static event handler

    void OnTextCtrlChanged(wxCommandEvent& Event);
    // void OnSliderCtrlChanged(wxCommandEvent& Event);
    void OnSliderCtrlChanged(wxScrollEvent & Event);
    void OnRadioButtonSelected(wxCommandEvent& Event);
    void OnRadioBoxSelected(wxCommandEvent& Event);
    void OnCheckBoxSelected(wxCommandEvent& Event);
    void OnComboBoxSelected(wxCommandEvent& Event);
    void OnListBoxSelected(wxCommandEvent& Event);
    void OnChoiceCtrlSelected(wxCommandEvent& Event);
    void OnButtonClicked(wxCommandEvent& Event);

    //-------------------------------------------------------------------------
    // specialized version of wxDialog::TransferDataToWindow() which disables
    // control change notifications while data is transferred to the controls
    // (otherwise, e.g. wxTextCtrl()->SetValue() would trigger a notification)
    virtual bool TransferDataToWindow();

    virtual void initializeDialog ();

    // execute validation with showing a message box for errors 
    // if ReturnCode != -1, EndDialog(ReturnCode) is called if
    // validation was successful
    bool ValidateWithMessageBox (int ReturnCode = -1);

    //-------------------------------------------------------------------------

protected:
    wxMultiLineText m_ErrorMessageMultiLineText;
    wxStatusMessageTarget * m_StatusMessageTarget;

    int m_DialogFontSize;
    int m_DialogFontHeight;

    bool m_IsSetFontIfSameAsDefaultSize;

    wxSize m_DefaultButtonSize;
    wxSize m_DefaultTextCtrlSize;
    wxSize m_DefaultListBoxSize;
    wxSize m_DefaultChoiceCtrlSize;
    wxSize m_DefaultComboBoxSize;
    double m_ZoomValue;
    double m_DialogZoomValue;
    static double m_GlobalDialogZoom;
    wxSize m_DesktopReferenceSize;
    static wxSize m_GlobalDesktopReferenceSize;
    double m_BorderCompressionValue;

    bool m_IsWatchedControlChangeNotification;
    wxObjectPtrArray m_WatchedObjectArray;
    // m_IsDialogInitialized: flag to let control change notification 
    // appear only after the dialog is initialized:
    bool m_IsDialogInitialized;

    // DECLARE_EVENT_TABLE()

    friend class wxExtDialog;
};

//-------------------------------------------------------------------------

class wxExtDlgBaseEvtHandler : public wxEvtHandler
{
    DECLARE_CLASS(wxExtDlgBaseEvtHandler)
 public:

    wxExtDlgBaseEvtHandler() { m_ExtDlgBase = NULL; }
    ~wxExtDlgBaseEvtHandler();
    wxExtDlgBaseEvtHandler(wxExtDlgBase * ExtDlgBase) { m_ExtDlgBase = ExtDlgBase; }
    void setExtDlgBase (wxExtDlgBase * ExtDlgBase) { m_ExtDlgBase = ExtDlgBase; }

    //-------------------------------------------------------------------------

    void OnCloseEvent (wxCloseEvent & CloseEvent) { m_ExtDlgBase -> OnCloseEvent(CloseEvent); }

    //-------------------------------------------------------------------------
    // notification functions called by the static event handler

    void OnTextCtrlChanged(wxCommandEvent& Event) { m_ExtDlgBase -> OnTextCtrlChanged(Event); }
    // void OnSliderCtrlChanged(wxCommandEvent& Event) { m_ExtDlgBase -> OnSliderCtrlChanged(Event); }
    void OnSliderCtrlChanged(wxScrollEvent & Event) { m_ExtDlgBase -> OnSliderCtrlChanged(Event); }
    void OnRadioButtonSelected(wxCommandEvent& Event) { m_ExtDlgBase -> OnRadioButtonSelected(Event); }
    void OnRadioBoxSelected(wxCommandEvent& Event) { m_ExtDlgBase -> OnRadioBoxSelected(Event); }
    void OnCheckBoxSelected(wxCommandEvent& Event) { m_ExtDlgBase -> OnCheckBoxSelected(Event); }
    void OnComboBoxSelected(wxCommandEvent& Event) { m_ExtDlgBase -> OnComboBoxSelected(Event); }
    void OnListBoxSelected(wxCommandEvent& Event) { m_ExtDlgBase -> OnListBoxSelected(Event); }
    void OnChoiceCtrlSelected(wxCommandEvent& Event) { m_ExtDlgBase -> OnChoiceCtrlSelected(Event); }
    void OnButtonClicked(wxCommandEvent& Event) { m_ExtDlgBase -> OnButtonClicked(Event); }

 protected:
    wxExtDlgBase * m_ExtDlgBase;

 public:
    DECLARE_EVENT_TABLE()     
    DECLARE_NO_COPY_CLASS(wxExtDlgBaseEvtHandler)
};

//-------------------------------------------------------------------------

class wxExtDlgHelper : public wxExtDlgBase
{
 public: 
    wxExtDlgHelper();
    ~wxExtDlgHelper();
    void setup (wxWindow * Window);
    void release (wxWindow * Window);

    virtual void setDialog (wxWindow * Window) { m_Window = Window; }
    virtual wxWindow * getDialog() { return m_Window; }

 protected:
    wxWindow * m_Window;
    wxExtDlgBaseEvtHandler m_ExtDlgBaseEvtHandler;
};

//-------------------------------------------------------------------------

// wxExtDialog:
// - wxDialog plus all functions from wxExtDlgBase class
class wxExtDialog : public wxDialog, public wxExtDlgBase
{
    DECLARE_DYNAMIC_CLASS(wxExtDialog)
 public:
    wxExtDialog ();

    wxExtDialog(wxWindow *parent, wxWindowID id,
              const wxString& title,
              const wxPoint& pos = wxDefaultPosition,
              const wxSize& size = wxDefaultSize,
              long style_DIALOG_STYLE = wxDEFAULT_DIALOG_STYLE,
              const wxString& name = wxDialogNameStr);

    virtual wxWindow * getDialog ();

    virtual void InitDialog ();

    ~wxExtDialog ();

    //-------------------------------------------------------------------------

    void OnCloseEvent (wxCloseEvent & CloseEvent) { wxExtDlgBase::OnCloseEvent(CloseEvent); }

    //-------------------------------------------------------------------------
    // notification functions called by the static event handler

    void OnTextCtrlChanged(wxCommandEvent& Event) { wxExtDlgBase::OnTextCtrlChanged(Event); }
    // void OnSliderCtrlChanged(wxCommandEvent& Event) { wxExtDlgBase::OnSliderCtrlChanged(Event); }
    void OnSliderCtrlChanged(wxScrollEvent & Event) { wxExtDlgBase::OnSliderCtrlChanged(Event); }
    void OnRadioButtonSelected(wxCommandEvent& Event) { wxExtDlgBase::OnRadioButtonSelected(Event); }
    void OnRadioBoxSelected(wxCommandEvent& Event) { wxExtDlgBase::OnRadioBoxSelected(Event); }
    void OnCheckBoxSelected(wxCommandEvent& Event) { wxExtDlgBase::OnCheckBoxSelected(Event); }
    void OnComboBoxSelected(wxCommandEvent& Event) { wxExtDlgBase::OnComboBoxSelected(Event); }
    void OnListBoxSelected(wxCommandEvent& Event) { wxExtDlgBase::OnListBoxSelected(Event); }
    void OnChoiceCtrlSelected(wxCommandEvent& Event) { wxExtDlgBase::OnChoiceCtrlSelected(Event); }
    void OnButtonClicked(wxCommandEvent& Event) { wxExtDlgBase::OnButtonClicked(Event); }

#if wxCHECK_VERSION (2, 8, 0)
    void OnOK(wxCommandEvent & Event);
    void OnCancel(wxCommandEvent & Event);
#endif

    //-------------------------------------------------------------------------
    // specialized ShowModal() function which
    // centers the dialog before displaying it
    virtual int ShowModal ();

    // EndDialog() calls EndModal() for modal, Hide() for non-modal dialogs
    // (same as function EndDialog() defined in wxDialog, but with public 
    // access qualifier)
    virtual void EndDialog(int ReturnCode);

    // multiple inheritance ambiguity resolution
    virtual bool TransferDataToWindow() { return wxExtDlgBase::TransferDataToWindow(); }

public:

    DECLARE_EVENT_TABLE()     
};

//-------------------------------------------------------------------------

extern bool ValidateWithMessageBox(wxWindow * Window, wxStatusMessageTarget * StatusMessageTarget);
extern bool ValidateWithMessageBox(wxDialog * Dialog, wxStatusMessageTarget * StatusMessageTarget, 
                                   int ReturnCode);

//-------------------------------------------------------------------------

class wxSizerAdder
{
public:
    wxSizerAdder ();
    virtual ~wxSizerAdder () { };

    wxSizerAdder (wxSizer * Sizer);
    wxSizerAdder (wxSizer * Sizer, int SizerColumnCount, int SizerRowCount);
#ifdef M_IsHaveGridBagSizer
    wxSizerAdder (wxGridBagSizer * GridBagSizer, int XPos, int YPos, int XSpan, int YSpan);
#endif

    void init (wxSizer * Sizer);
    void init (wxSizer * Sizer, int SizerColumnCount, int SizerRowCount);
#ifdef M_IsHaveGridBagSizer
    void init (wxGridBagSizer * GridBagSizer, int XPos, int YPos, int XSpan, int YSpan);
#endif

    void setGroupSize (int SizerGroupSize = -1);

    void nextGroup();

    void addWindow (wxWindow * Window, int Flags, int Border);
    void addSizer (wxSizer * Sizer, int Flags, int Border);
    void addSpace (int Width, int Height, int Flags, int Border);

    wxSizer * getSizer();
    wxSizer * getSizerConst() const;

    // ParentWindow is simply a variable that may be read by wxExtDlgBase
    // control adder functions
    void setParentWindow (wxWindow * ParentWindow);
    wxWindow * getParentWindow () const;

    void setColumnGrowable (int ColumnIndex, bool IsGrowable = true);
    void setRowGrowable (int RowIndex, bool IsGrowable = true);
    void setAllGrowable();

private:
    void init();

    wxSizer * m_Sizer;
#ifdef M_IsHaveGridBagSizer
    wxGridBagSizer * m_GridBagSizer;
#endif
    int m_CurrentCount;
    int m_SizerGroupSize;

    int m_SizerColumnCount;
    int m_SizerRowCount;

    int m_GridBagXPos;
    int m_GridBagYPos;
    int m_GridBagXSpan;
    int m_GridBagYSpan;

    wxWindow * m_ParentWindow;
};

//-------------------------------------------------------------------------

// wxSizerAdderProxy
// - proxy class for wxExtDialog compatibility: allow wxSizer* arguments
//   to wxExtDialog control addition helper functions, implicit creation 
//   of wxBoxSizer if Sizer == NULL
class wxSizerAdderProxy
{
 public:
    wxSizerAdderProxy ();
    wxSizerAdderProxy (wxSizerAdder * SizerAdder);
    wxSizerAdderProxy (wxSizer * Sizer);

    wxSizerAdder * makeSizerAdder(wxSizerAdder & AlternativeSizerAdder) const;

 private:
    wxSizerAdder * m_SizerAdderPtr;
    wxSizer * m_SizerPtr;
};

//-------------------------------------------------------------------------

#define M_DeclareParameter(ParameterName,ParameterType) \
  public:                                                      \
    void set##ParameterName (const ParameterType & ParameterName) \
    { \
      m_##ParameterName = ParameterName; \
      m_Is##ParameterName##Initialized = true; \
    } \
    const ParameterType & get##ParameterName () const \
    { \
      return m_##ParameterName; \
    } \
    bool getIs##ParameterName##Initialized () const \
    { \
      return m_Is##ParameterName##Initialized; \
    } \
    void set##ParameterName##InitializedState (bool IsInitialized = true) \
    { \
      m_Is##ParameterName##Initialized = IsInitialized; \
    } \
 private: \
    ParameterType m_##ParameterName; \
    bool m_Is##ParameterName##Initialized \


//-------------------------------------------------------------------------

// wxWindowAdderDefaultParameterSet
// - these parameters are used to customize the various functions of 
//   wxWindowAdder 
// - example:
//   > WA.accessDefaults().setSizerBorder (_B(10));
//   > WA.accessDefaults().setLabelFontColour (wxColour (*wxBLUE));
//   where 'WA' is an instance of wxWindowAdder class
class wxWindowAdderDefaultParameterSet
{
 public:
    wxWindowAdderDefaultParameterSet();
    virtual ~wxWindowAdderDefaultParameterSet();
    void resetToDefaults (wxExtDlgBase * ExtDlgBase);

    // set both, horizontal and vertical spacing:
    void setSizerSpacing (int SizerSpacing);

    // NOTE currently, IsDebugWarningOnShortcutClash only works if
    // also IsFixLabelStringOnShortcutClash is enabled
    M_DeclareParameter (IsFixLabelStringOnShortcutClash, bool);
    M_DeclareParameter (IsDebugWarningOnShortcutClash, bool);

    M_DeclareParameter (SizerHSpacing, int);
    M_DeclareParameter (SizerVSpacing, int);
    M_DeclareParameter (SizerBorder, int);

    M_DeclareParameter (WindowFontColour, wxColour);
    M_DeclareParameter (WindowBackgroundColour, wxColour);

    M_DeclareParameter (LabelFontColour, wxColour);
    M_DeclareParameter (LabelBackgroundColour, wxColour);
    M_DeclareParameter (PostTextFontColour, wxColour);
    M_DeclareParameter (PostTextBackgroundColour, wxColour);

    M_DeclareParameter (TransparentStaticTextFlags, wxTransparentStaticText::ETransparentStaticTextFlags);

    M_DeclareParameter (IsWatchControl, bool);
};

//-------------------------------------------------------------------------

#undef M_DeclareParameter

//-------------------------------------------------------------------------

class wxOverlayAdder
{
 public:
    enum EMode
      {
        IsAddOnPop = 0x0001,
        ZeroModeBits = 0
      };

    wxOverlayAdder ();
    void init();

    wxWindow * m_OverlayWindow;
    wxWindow * m_PreviousParentWindow;
    wxSizer * m_OverlaySizer;
    EMode m_ModeBits;
};


//-------------------------------------------------------------------------

WX_DECLARE_OBJARRAY(wxWindowAdderDefaultParameterSet, wxWindowAdderDefaultParameterSetArray);

WX_DECLARE_OBJARRAY(wxSizerAdder, wxExtLibSizerAdderArray);

WX_DECLARE_OBJARRAY(wxOverlayAdder, wxOverlayAdderArray);

//-------------------------------------------------------------------------

// wxWindowAdder
// - simplify handling sizers: maintains a stack of sizers which
//   can be manipulated with push and pop functions
// - simplify creation of labels
class wxWindowAdder
{
public:
    wxWindowAdder();

    //-------------------------------------------------------------------------
    //-- ampersand-shortcut clash detection helper
    //
    // setDetectShortcutClashs(), checkAndFixLabelString()
    // - in debug and release mode, a duplicated short-cut is disabled 
    //   simply by removing the ampersand character 
    //   (unless parameter IsFixLabelStringOnShortcutClash is false)
    // - only in debug mode: assert message dialog is shown in case of clash
    //   (unless parameter IsDebugWarningOnShortcutClash is false)
    // - duplicated short-cuts can be confusing or worse: if, in the same 
    //   dialog, a label of some text-control is named '&domain', and a button 
    //   is named '&Delete', it just depends on the current focus if either
    //   text-control is activated, or '&delete' button gets a click event!
    //   (scope: WinXP, probably other window systems too)
    // - default: short-cut clash detection enabled
    // 
    // note:
    // - checkAndFixLabelString() could try to automatically assign a different
    //   yet unused short-cut, but this may cause other short-cuts
    //   to be disabled/"moved" then, so this is not really desireable
    // - the function could be modified to remove all short-cuts,
    //   if this is needed in some context (e.g. for some window manager
    //   that doesn't support the ampersand short-cuts)
    //
    // see also:
    // - related functions for menu items exist:
    //     wxMenuItem::GetLabelFromText(), ::wxStripMenuCodes()
    //
    void checkAndFixLabelString (const wxString & InputLabelString, 
                                 wxString & CheckedLabelString,
                                 bool IsAlwaysRemoveAmpersand = false);

    //-------------------------------------------------------------------------
    // helper functions to make creation of dialog controls easier
    // - for notes on size specifications with regard to zooming, see
    //   the getAdjusted*Width/Height functions

    void setExtDlgBase (wxExtDlgBase * ExtDlgBase);
    // for backwards-compat.:
    void setDialog (wxExtDialog * ExtDialog);
    void setParentWindow (wxWindow * ParentWindow);

    void setSize (const wxSize & Size = wxDefaultSize);
    void setSize (int Width, int Height);

    // window style used for functions without Style parameter
    // (note that some functions have window style parameters with a 
    // default)
    void setWindowStyle (long WindowStyle);

    void setLabel (const wxString & LabelString);
    const wxString & getLabel();
    void setPostText (const wxString & PostTextString);
    const wxString & getPostString();

    void pushGroup (const wxString & GroupNameString);
    void popGroup ();

    wxString getFullName ();

    void setValidator (const wxValidator & Validator);

    void addSpace (int Width, int Height);
    void addHSpace (int Width);
    void addVSpace (int Height);

    void setSizerHSpacing(int Width);
    void setSizerVSpacing(int Height);

    void skipCell ();
    void skipCells (int Count);
    void nextGroup();
    void next(); // alias for nextGroup()

    wxWindow * getWindow();
    wxSizer * getSizer();
    wxSizerAdder & getSizerAdder();

    //-------------------------------------------------------------------------
    void addSizer (wxSizer * Sizer, int AddFlags, int AddBorder);
    void pushSizer (wxSizer * Sizer);
    void addAndPushSizer (wxSizer * Sizer, int AddFlags, int AddBorder);
    void popSizer ();

    //-------------------------------------------------------------------------
    void addAndPushFlexGridSizer (int Columns, bool IsAddBorder = false, bool IsAddOnlySizer = true);
    void addAndPushStaticBoxAndSizer (const wxString & LabelString,
                                      wxSizer * SubSizer,
                                      int AddFlags, int AddBorder);
    void addAndPushStaticBoxAndFlexGridSizer (const wxString & LabelString, int ColumnCount,
                                              bool IsAddBorder = false);

    //-------------------------------------------------------------------------
    void pushDefaults();
    void popDefaults();
    const wxWindowAdderDefaultParameterSet & getDefaults();
    wxWindowAdderDefaultParameterSet & accessDefaults();

    //-------------------------------------------------------------------------
    // overlay support (put controls onto other control (e.g. background image,
    // notebook page etc.)
    void pushOverlay (wxWindow * OverlayWindow, wxOverlayAdder::EMode Mode);
    void popOverlay ();

    //-------------------------------------------------------------------------

    wxWindow * getAddParentWindow ();

    wxWindow * addWindow (wxWindow * Window);

    // DEF ButtonPtrPtr can be NULL
    // DEF ButtonId can be -1 to obtain an automatically generated button id
    void addButton (const wxString & LabelString,
                    wxButton ** ButtonPtrPtr, wxWindowID CtrlIdent,
                    wxCommandEventFunction CommandEventFunction, 
                    long Style = 0);
    void addButton (const wxString & LabelString,
                    wxCommandEventFunction CommandEventFunction,
                    long Style = 0);
    // removed: param long Style = 0; to allow overload with wxString, wxWindowID
    void addButton (const wxString & LabelString); 
    void addButton (const wxString & LabelString, wxWindowID CtrlIdent);
    void setDefaultButton();

    void addCheckBoxWithLabel (wxCheckBox ** CheckBoxPtrPtr,
                               const wxString & LabelString,
                               long Flags = 0,  // Flags can be 0 or wxTE_READONLY
                               bool IsLabelOnTheLeft = false);
    void addRadioButtonWithLabel (wxRadioButton ** RadioButtonPtrPtr,
                                  const wxString & LabelString,
                                  long RadioCtrlStyle = 0);

    // TODO add sizer flags for textctrl, e.g. Expandable TextCtrl, ExtraSizer,
    void addTextCtrl (wxTextCtrl ** TextCtrlPtrPtr,
                      long TextCtrlStyle = wxTE_LEFT);

    void addLabel (wxStaticText ** StaticTextPtrPtr,
                   const wxString & LabelString,
                   long WindowStyle = wxALIGN_LEFT);

    void addMultiLineText (wxMultiLineText * MultiLineText,
                           long CtrlStyle = 0,
                           int MultiLineTextStyle = 0);
    void addMultiLineText (wxMultiLineText * MultiLineText,
                           long CtrlStyle,
                           int MultiLineTextStyle,
                           const wxString & TextString,
                           int MinLineCount, int MaxLineCount,
                           int BorderWidth=0);

    void addListBox (wxListBox ** ListBoxPtrPtr, 
                     int ChoiceCount, const wxString ChoiceStringArray []);
    void addChoiceCtrl (wxChoice ** ChoiceCtrlPtrPtr, 
                        int ChoiceCount, const wxString ChoiceStringArray []);
    void addComboBox (wxComboBox ** ComboBoxPtrPtr, 
                      int StringCount, const wxString StringArray []);

    void addStaticBitmap (wxStaticBitmap ** StaticBitmapPtrPtr,
                          const unsigned char * ImageData,
                          size_t ImageDataSize);
    void addStaticBitmap (wxStaticBitmap ** StaticBitmapPtrPtr,
                          const char * ResourceNameString);
    void addStaticBitmap (wxStaticBitmap ** StaticBitmapPtrPtr,
                          wxBitmap * Bitmap);

    void addStaticLine (long Style = wxLI_HORIZONTAL);

    void addHeadline (const wxString & HeadlineString);

    void watchControl();

protected:
    void createAndAddLabel ();
    void setCurrentWindow (wxWindow * Window);
    void createAndAddPostText ();

    void decorateWindow (wxWindow * Window);

    //-- shortcut clash detection helper
    bool m_IsDetectShortcutClashs;
    wxArrayString m_LabelStringArray;
    // for each label in m_LabelStringArray, m_ShortcutCharString contains
    // characters, which are either shortcut character (if found) or '\001' 
    // (a probably unused control character)
    wxString m_ShortcutCharString;

    // current values:
    wxString m_LabelString;
    wxString m_PostTextString;
    wxWindow * m_Window;
    wxExtDlgBase * m_ExtDlgBase;
    wxSize m_Size;
    long m_WindowStyle;

    bool m_IsLabelCreated;

    wxWindow * m_ParentWindow;

    wxExtLibSizerAdderArray m_SizerAdderArray;
    wxArrayString m_GroupNameStringArray;

    wxWindowAdderDefaultParameterSetArray m_WindowAdderDefaultParameterSetArray;
  
    wxOverlayAdderArray m_OverlayAdderArray;
};

//-------------------------------------------------------------------------

class wxWindowAdderAutoPopSizer
{
public:
    wxWindowAdderAutoPopSizer(wxWindowAdder * WindowAdder);
    wxWindowAdderAutoPopSizer(wxWindowAdder * WindowAdder, wxWindow * OverlayWindow,
                              wxOverlayAdder::EMode ModeBits);
    ~wxWindowAdderAutoPopSizer();
    void init();
    void reset();
    void pop();
    void push();

private:
    wxWindowAdder * m_WindowAdder;
    wxWindow * m_OverlayWindow;
    wxOverlayAdder::EMode m_OverlayAdderModeBits;
};

#if 0
class wxWindowAdderPopHandler
{
 public:
    virtual void pop () = 0;
};

class wxNotebookPageWindowAdderPopHandler
{
 public:
    virtual void pop ();
 protected:
    wxWindow * m_PreviousParentWindow;
    // maybe:
    wxString m_NotebookPageName;
};
#endif

// example of wxWindowAdderAutoPopSizer usage in macro:
//   #define SIZER(WindowAdder,AddAndPushCall)    
//      WindowAdder. AddAndPushCall ;
//      wxWindowAdderAutoPopSizer WindowAdderAutoPopSizer__ (& WindowAdder);
//
//   {
//      SIZER(addAndPushSizer (new wxBoxSizer (wxVERTICAL),
//                             wxLEFT|wxRIGHT|wxTOP|wxEXPAND, _B(0)));
//      ... add controls here
//   }
//   {
//      SIZER(addAndPushSizer (new wxBoxSizer (wxVERTICAL),
//                             wxLEFT|wxRIGHT|wxTOP|wxEXPAND, _B(0)));
//      ... add controls here
//   }    

//-------------------------------------------------------------------------

class wxNonModalDialogInfo
{
 public:
    wxNonModalDialogInfo();

    long m_CommandIdent;
    wxWindow * m_Window;

    // maybe: long (or int) m_ExitCode
    // - would require some extra function in wxNonModalDialogManag
    //   which would be used in place of EndModal()/EndDialog()
    // - there was currently no need for keeping some dialog
    //   exit code
};

WX_DECLARE_OBJARRAY(wxNonModalDialogInfo, wxNonModalDialogInfoArray);

//-------------------------------------------------------------------------

// wxNonModalDialogManag
// - provide unified handling for non-modal dialogs
// - command idents need to be registered
// - dialog/window creation is done through a callback function
// - a pointer to opened dialog window instance is kept
//   for each registered dialog (i.e. command ident), which 
//   was created through this class
// - default handling for dialog show menu entries
class wxNonModalDialogManag
{
 public:
    wxNonModalDialogManag();
    virtual ~wxNonModalDialogManag();

    // handleMenuCommand()
    // - default handling for dialog show/hide menu commands
    //   - if not yet created: create dialog through 
    //     createWindow() callback 
    //   - if dialog is created but hidden: show dialog and bring to
    //     front
    void handleMenuCommand (long CommandIdent);

    // closeAll():
    // - implementation note: uses special hack to close hidden windows
    //   by showing them before closing (otherwise, application may not
    //   close properly in some wx ports)
    // - todo: support vetoing
    void closeAll();
  
    // callback to create windows/dialogs (needs to be overwritten)
    // - this allows the callback to create dialog instance
    //   by using desired class and constructor arguments
    virtual wxWindow * createWindow (long CommandIdent) = 0;
  
    //-- info array functions
    // addInfo(): register CommandIdent
    int addInfo (long CommandIdent);
    int findInfo (long CommandIdent);
    int findInfoFromWindow (wxWindow * Window);
    wxNonModalDialogInfo & getInfoByIndex (int InfoIndex);

 protected:
    wxNonModalDialogInfoArray m_NonModalDialogInfoArray;
};

//-------------------------------------------------------------------------

// wxDisablingBusyInfo
// - wxBusyInfo calls wxYield(), which may cause input events to be processed.
//   In many cases nested handling of input events is not not desirable, 
//   so use wxWindowDisabler() to disable processing input events while
//   showing the wxBusyInfo.
// - under wxMSW, use of wxWindowDisabler seems to cause loosing the
//   focus or activation of previously focused or activated window, so
//   to avoid this, wxFocusRestorer is used
class wxDisablingBusyInfo
{
public:
    wxDisablingBusyInfo (const wxString & MessageString, wxWindow * ParentWindow = NULL,
                         wxWindow * SkipDisableWindow = NULL);

private:
    wxFocusRestorer m_FocusRestorer;
    wxWindowDisabler m_WindowDisabler;
    wxBusyInfo m_BusyInfo;
};

//-------------------------------------------------------------------------

class wxCalendarCtrl;
class wxCalendarEvent;

// wxCalendarDialog
// - dialog which contains wxCalendarCtrl and wxTraitedDateTimeValidator
//   to allow input and validation of single wxDateTime value
// - both date input controls try to mutually update each other
// - currently only the date part can be selected, time part is set to 
//   zero
class wxCalendarDialog : public wxExtDialog
{
  public:
    // - default for DateTimeFormatString: as for wxTraitedDateTimeValidator
    wxCalendarDialog (wxWindow * ParentWindow,
                      const wxString & DialogTitleString,
                      wxDateTime * DateTimePtr,
                      const wxDateTime & MinDateTime, const wxDateTime & MaxDateTime,
                      long DateTimeValidatorFlags,
                      const wxString & DateTimeFormatString = wxEmptyString);
    void init();
    void addStatusMessageTarget (wxWindowAdder & WA);
    void clearControlPointers();
    void createControlsMain ();
    void createControls (wxWindowAdder & WA);
    void fillControlsPostCreate ();

    void OnCalendarDoubleClicked (wxCalendarEvent & Event);
    void OnCalendarSelectionChanged (wxCalendarEvent & Event);
    void handleSelectionChange(wxCalendarEvent & Event);

    void OnOkayButton (wxCommandEvent & Event);
    void handleAcceptInput();

    wxTraitedDateTimeValidator & accessDateTimeValidator();

 private:
    long m_DateTimeValidatorFlags;
    wxDateTime m_MinDateTime;
    wxDateTime m_MaxDateTime;
    wxString m_DateTimeFormatString;

    wxMultiLineText m_StatusMessageMultiLineText;
    wxCalendarCtrl * m_DateCalendarCtrl;
    wxTextCtrl * m_DateTimeTextCtrl;

    // m_TextCtrlDateTime is used to hold a copy of currently selected
    // date (which is also displayed in the read-only TextCtrl then)
    wxDateTime m_TextCtrlDateTime;
    wxDateTime * m_DateTimePtr;
};

void showCalendarDialog (wxWindow * ParentWindow,
                         const wxString & CalendarDialogCaptionString,
                         const wxDateTime & MinDateTime,
                         const wxDateTime & MaxDateTime,                         
                         long DateTimeValidatorFlags,
                         const wxString & DateTimeFormatString,
                         wxTextCtrl * DateTimeTextCtrl, 
                         bool IsCallResetTime = false);

//-------------------------------------------------------------------------

class wxNonModalDialogManag;

enum ETopLevelWindowModification
  {
    HideTLWModification = 1,
    ShowAndRaiseTLWModification
  };

// modifyTopLevelWindows()
// - OBSOLETE extended functionality is now in wxAppRootFrame class
// - helper functions for hidden application root frame support, 
//   this was a first try only and does not do everything that is 
//   desired (e.g. keeping Z-order of windows)
// - this function however, maybe used for other purposes in the future
void modifyTopLevelWindows (wxWindow * ParentWindow, wxWindow * ExcludeWindow, ETopLevelWindowModification Mode,
                            wxNonModalDialogManag * NonModalDialogManag,
                            wxWindow * ExtraWindow);

//-------------------------------------------------------------------------

// TLWStateInfo
// - class to save state info on wxTopLevelWindow (wxTopLevelWindowNative)
//   windows
class wxTLWStateInfo
{
 public:
    wxTLWStateInfo();
    wxTLWStateInfo (const wxTLWStateInfo & TLWStateInfo);
    wxTLWStateInfo (wxWindow * Window, bool IsShownWhenNotIconized);

    // Shown/Hidden state at the time of iconization request:
    bool m_IsShownWhenNotIconized;

    // pointer to window:
    wxWindow * m_Window;
};

WX_DECLARE_OBJARRAY(wxTLWStateInfo, wxTLWStateInfoArray);

//-------------------------------------------------------------------------

#if (M_WxExtLib_IsUseAppRootFrame == 1)

// wxAppRootFrame
// - purpose: helper class to allow the application main frame window 
//   (containing menu etc.) to be raised to the foreground, before
//   other top-level windows (TLW) (e.g. non-modal dialogs)
// - this is useful if non-modal dialogs are used, which would 
//   otherwise be always in front of the main frame window
//   (normal behavior of child window having popup style)
// - following structure is intended:
//      wxAppRootFrame
//         \                                                 //
//          |- AppSpecificMainFrame
//          |- NonModalDialog1
//          |- ...
//          |- NonModalDialogN
// - windows of this class are normally never shown
//   - however, because of hidden state, iconize/restore behavior needs to
//     be implemented manually (which this class tries to emulate)
//   - otherwise a small dummy window could be used, but this might
//     have other side effects when e.g. user does a "cascade" or other 
//     alignment of top-level/application windows
// - there should be only one instance of this class per process
// - notation: here, the term TLW is used to denote frame windows and non-modal 
//   dialogs that are children of an wxAppRootFrame instance
// - compatibility: these functions currently only work, more or less, as 
//   expected with wxMSW
//
class wxAppRootFrame : public wxFrame
{
    DECLARE_DYNAMIC_CLASS(wxAppRootFrame)
 public:
    wxAppRootFrame ();

#if wxCHECK_VERSION (2, 5, 0)
    typedef wxTopLevelWindowNative wxTopLevelType;
#else
    typedef wxWindow wxTopLevelType;
#endif

    wxAppRootFrame(wxWindow *parent, wxWindowID id,
                      const wxString& title,
                      const wxPoint& pos = wxDefaultPosition,
                      const wxSize& size = wxDefaultSize,
                      long style_DIALOG_STYLE = wxDEFAULT_DIALOG_STYLE,
                      const wxString& name = wxDialogNameStr);

    ~wxAppRootFrame ();

    void init();

    // ControllingFrame is the frame which contains the application main menu
    M_DeclarePtrProperty (wxFrame, ControllingFrame);

 public:
    // helper function: returns top-most top-level window (this should be
    // set as handler to replace NULL values for ParentWindow specification)
    static wxWindow * getDefaultParentWindow (wxWindow * ParentWindow);
  
    static wxAppRootFrame * create (const wxIcon & Icon);
    static void setup (wxFrame * FrameWindow);

    // handlers related to wxAppRootFrame's own window events
    void OnDestroyEvent (wxWindowDestroyEvent & DestroyEvent);
    void OnCloseEvent (wxCloseEvent & CloseEvent);
    void OnShowEvent (wxShowEvent & ShowEvent);
    void OnActivateEvent (wxActivateEvent & ActivateEvent);
    void OnActivateAppEvent (wxActivateEvent & ActivateEvent);
    void OnIconizeEvent (wxIconizeEvent & IconizeEvent);
    void OnMaximizeEvent (wxMaximizeEvent & MaximizeEvent);
    void OnMoveEvent (wxMoveEvent & MoveEvent);

    // handlers intended for top-level windows which are children of
    // this wxAppRootFrame
    // - notes:
    //   - not used: void OnTLWIconizeEvent (wxIconizeEvent & IconizeEvent);
    //   - CloseEvent and ShowEvent are not used either
    //   - IMPORTANT: these functions are really called with a different 'this' pointer
    //     (due to the way the event handler is added using wxEvtHandler::Connect())
    //     and only act as proxy which forward to instances of this class
    //     via s_GlobalAppRootFrame pointer)
    //     - this technique may not be guaranted to work with any C++ compiler or 
    //       by the C++ standard, but probably will work with the binary layout of 
    //       most compilers
    //     - the method has the advantage that it can be applied without adding
    //       these handlers to the TLW window classes itself, and thus works
    //       even for classes from 3rd parties etc.
    //     - alternatively, it would be required to put the forwarding 
    //       event handlers in all used TLW classes
    void OnTLWDestroyEventForward (wxWindowDestroyEvent & DestroyEvent);
    void OnTLWCloseEventForward (wxCloseEvent & CloseEvent);
    void OnTLWShowEventForward (wxShowEvent & ShowEvent);
    void OnTLWActivateEventForward (wxActivateEvent & ActivateEvent);
    void OnTLWIconizeEventForward (wxIconizeEvent & IconizeEvent);
    void OnTLWMaximizeEventForward (wxMaximizeEvent & MaximizeEvent);
    void OnTLWMoveEventForward (wxMoveEvent & MoveEvent);

    //-- TLW event handling
    // - some of the event types are not yet used specially (e.g. ShowEvent)
    //   may not require special handling
    // - except for notifyActivateAppEvent(), connectTLWEventHandlers() adds
    //   handlers which forward to the following functions
    // - notifyActivateAppEvent() must be forwarded manually from an
    //   wxApp::SetActive() overload (see file wxWidgets-base/include/wx/app.h 
    //   for declaration)
    bool notifyActivateAppEvent (bool IsActivate, wxWindow * LastFocusWindow);
    bool notifyTLWDestroyEvent (wxWindowDestroyEvent & DestroyEvent);
    bool notifyTLWCloseEvent (wxCloseEvent & CloseEvent /*, bool IsHideInsteadOfClose */);
    bool notifyTLWShowEvent (wxShowEvent & ShowEvent);
    // as an exception, notifyTLWActivateEvent() is also used to handle 
    // activate events of wxAppRootFrame
    bool notifyTLWActivateEvent (wxActivateEvent & ActivateEvent);
    bool notifyTLWIconizeEvent (wxIconizeEvent & IconizeEvent);
    bool notifyTLWMaximizeEvent (wxMaximizeEvent & MaximizeEvent);
    bool notifyTLWMoveEvent (wxMoveEvent & MoveEvent);

    // connectTLWEventHandlers()
    // - this needs to be called for each managed top-level window once
    // - this should be called after window creation and before the first
    //   Show(true) of the Window, so the window is inserted in the list 
    //   of TLWs at first wxEVT_ACTIVATE event
    void connectTLWEventHandlers (wxWindow * Window);

    // obtain wxTopLevelWindowNative pointer by use of wxDynamicCast(), or return
    // NULL if not convertible
    bool getTLWFromEvent (wxEvent & Event, wxTopLevelType ** TLWNativePtrPtr);

    //-- functions to manipulate list of top-level windows
    int pushInfo (const wxTLWStateInfo & TLWStateInfo);
    void deleteInfo (int Index);
    void deleteInfoFromWindow (wxWindow * Window);
    wxTLWStateInfo & getInfoByIndex (int InfoIndex);
    int findInfoFromWindow (wxWindow * Window);

    //-- helper variables
    // - ignore recursive events which would further complicate issues
    bool m_IsLockEventProcessing;
    // - ignore multiple same events (if an application is activated, more two
    //   events indicating this may be generated, for whatever reason (scope:
    //   wxMSW)
    bool m_IsFirstActivateAppEvent;
    wxWindow * m_LastFocusWindow;

 protected:
    //-- list of top-level windows: this list is used to:
    // - raise windows after iconization in correct Z-order
    // - save show/hide state at the moment of iconization (so upon restore
    //   after iconization, Show() and Raise() is called only for the windows 
    //   which were shown before)
    // notes:
    // - Z-order handling is done without using OS-specific functions
    //   like ::SetWindowPos() in MSW
    //   - upon a wxEVT_ACTIVATE event, the window is brought to foreground
    //     and this is reflected by putting it at the top(end) of the 
    //     array/stack
    wxTLWStateInfoArray m_TLWStateInfoArray;

 public:
    DECLARE_EVENT_TABLE()     
};

#else

typedef wxWindow wxAppRootFrame;

#endif // M_WxExtLib_IsUseAppRootFrame

// return global singular instance of wxAppRootFrame (or NULL)
wxAppRootFrame * wxGetGlobalAppRootFrame();

//-------------------------------------------------------------------------

#define M_IsUndefExportMagic 1
#    include "WxExtLibAliases.h"
#undef M_IsUndefExportMagic

//-------------------------------------------------------------------------

#endif
