// -*- c++ -*-
/*
-------------------------------------------------------------------------
This file is part of WxWidgetsExtensions library.
-------------------------------------------------------------------------

WxExtLib (WxWidgetsExtensions) library
-----------------------------

COPYRIGHT NOTICE:

WxExtLib library Copyright (c) 2003-2007 Daniel Kps

The WxWidgetsExtensions library and associated documentation files (the
"Software") is provided "AS IS".  The author(s) disclaim all
warranties, expressed or implied, including, without limitation, the
warranties of merchantability and of fitness for any purpose.  The
author(s) assume no liability for direct, indirect, incidental,
special, exemplary, or consequential damages, which may result from
the use of or other dealings in the Software, even if advised of the
possibility of such damage.

Permission is hereby granted, free of charge, to any person obtaining
a copy of this Software, to deal in the Software without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Software,
and to permit persons to whom the Software is furnished to do so,
subject to the following conditions:

 1. The origin of this source code must not be misrepresented.
 2. Altered versions must be plainly marked as such and must not be
    misrepresented as being the original source.
 3. This Copyright notice may not be removed or altered from any 
    source or altered source distribution.

End of WxExtLib library Copyright Notice

-------------------------------------------------------------------------

RATIONALE

1) programming related:
   a) use of config entry objects brings more security against
      typing mistakes of section/key name strings
   b) type safety at the level of accessing the config variables
      via config-entry objects
   c) names of access can differ from names in the config, this way
      we could easily keep legacy names in the config while using
      other possibly better names in the program to access them
   d) default values can be centralized in config entry objects and
      therefore duplication of default values can be avoided

2) runtime related:
   if all entries are stored in a list, we can:
   a) enumerate all entries and write default values for each
      to the config at once, so the user can have an overview of
      all sections/keys and their defaults
   b) write all values at once (e.g. upon program termination)

   moreover, we could:
   c) check the range of e.g. integer and long variables,
      and we could open a message box/or return a string containing
      information about invalid entries
   d) implement undefined default value handling if 
      section/key value is empty
   e) read-cache entries: e.g. don't reget an entry if it was
      read no more than 500 msec before
   f) write-cache entries
   g) write entries only if a change is detected, or if some
      modification flag is set,
      - to speed up writing
      - to avoid overwriting changes made by the user while the
        program is running

IMPLEMENTATION

TODO

*/
//-------------------------------------------------------------------------

#ifndef _INCLUDED_ConfigEntry_h  
#define _INCLUDED_ConfigEntry_h  

#include "WxExtLibConfig.h"

#if defined(__GNUG__) && (!defined(__APPLE__)) && (!defined(M_NoPragmaInterface))
#   pragma interface "ConfigEntry.h"
#endif

#include <wx/defs.h>
#include <wx/confbase.h>
#include <wx/datetime.h>

#include "WxMisc.h"

//-------------------------------------------------------------------------

#define M_IsDefineExportMagic 1
#    include "WxExtLibAliases.h"
#undef M_IsDefineExportMagic

//-------------------------------------------------------------------------

class wxConfigEntry;

WX_DEFINE_ARRAY_PTR(wxConfigEntry *, CConfigEntryArray);

class wxConfigEntryManager
{
 public:
    wxConfigEntryManager();
    ~wxConfigEntryManager();

    void addConfigEntry (wxConfigEntry * ConfigEntry);

    // NOTE currently, setAssociatedConfig() must be called before
    // calls to addConfigEntry(). It won't have an effect for previously
    // added wxConfigEntry objects
    void setAssociatedConfig (wxConfigBase * ConfigBase);
    wxConfigBase * getAssociatedConfig ();

    // this reads all entries (not that an entry is re-read only if
    // some time passed after last read and if it was not modified
    // in the memory)
    bool readAllConfigEntries ();
    // writes all modified entries to the wxConfig object
    bool writeAllConfigEntries ();
    // writes all default entries
    bool setDefaultValues ();

 protected:
    CConfigEntryArray m_ConfigEntryArray;
    wxConfigBase * m_ConfigBase;
};

//-------------------------------------------------------------------------

// DECLARE_ConfigEntry:
// - declaration of wxConfigEntry-type object together
//   with get and set accessors:
#define DECLARE_ConfigEntry(TValue, TConfigEntry,Name)      \
  TConfigEntry m_##Name;                                    \
  TValue get##Name()                                        \
    {                                                       \
      return m_##Name.get();                                \
    }                                                       \
  void get##Name(TValue & Value)                            \
    {                                                       \
      m_##Name.get(Value);                                  \
    }                                                       \
  void set##Name(const TValue & Value)                      \
    {                                                       \
      m_##Name.set(Value);                                  \
    }                                                       \

//-------------------------------------------------------------------------

// wxConfigEntry:
// Notes:
// - MessageString is intended to return information 
//   e.g. about missing or wrong entries (otherwise the user 
//   doesn't know if something is wrong because missing 
//   or wrong entries are silently replaced with the default);
//   (however, it is not yet used).
// TODO:
// - m_ReadMinimalTimeInterval should be made variable; also, it should
//   be possible to selectively deactivate this functionality
class wxConfigEntry
{
 public:
    wxConfigEntry ();
    virtual ~wxConfigEntry ();

    virtual void setEntryPath (const wxString & EntryPath);
    virtual void getEntryPath (wxString & EntryPath);

    virtual void setDontReread (bool IsDontReread);
    virtual void setDontWrite (bool IsDontWrite);
    virtual void setForcedWrite (bool IsForcedWrite);

    //-- functions called by wxConfigEntryManager:
    // write (write operation to config object is done
    // only if the entry was modified by the program or did
    // not exist in the wxConfig before)
    // - for MessageString, see class notes
    virtual bool write (wxString & MessageString) = 0;
    // read: an entry is re-read only if
    // - some time (see MinRereadIntervalMS)
    //   is passed after last read 
    // - and the value was not modified by the program via a set() 
    //   operation (see derived classed) and is not written yet
    //   (In this case, the value in the memory will override the
    //   value in the wxConfig).
    // - for MessageString, see class notes
    virtual bool read (wxString & MessageString) = 0;
    // set to the default value
    virtual void setToDefault () = 0;

    // setAssociatedConfig() will be called by wxConfigEntryManager
    // during wxConfigEntryManager::addConfigEntry()
    virtual void setAssociatedConfig (wxConfigBase * ConfigBase);

    //-- for use by derived classes:
    bool readInternal (wxString & String);
    bool readInternal (long & Long);
    bool readInternal (double & Double);
    bool readInternal (bool & Boolean);
    bool writeInternal (const wxString & String);
    bool writeInternal (long Long);
    bool writeInternal (double Double);
    bool writeInternal (bool Boolean);

    void setReadState ();
    bool getIsReadRequired ();
    void setWriteState ();
    bool getIsWriteRequired ();

 protected:
    // wxStopWatch m_LastReadTimeMS;
    wxMinimalTimeInterval m_ReadMinimalTimeInterval;

    // minimal time that must pass until an entry is actually re-read
    // from the wxConfig object
    enum { MinRereadIntervalMS = 1000 };

    // read, write and modification states:
    bool m_IsDontReread;
    bool m_IsDontWrite;
    bool m_IsForcedWrite;

    bool m_IsAlreadyRead;
    bool m_IsChangePending;
    bool m_IsSetPending;

    wxConfigBase * m_ConfigBase;
    wxString m_EntryPath;
};

//-------------------------------------------------------------------------

class wxIntegerConfigEntry : public wxConfigEntry
{
  public:
    wxIntegerConfigEntry();

    void init (const wxString & EntryPath,
               int MinInteger, int MaxInteger, int DefaultInteger);
    void setRangeAndDefault (int MinInteger, int MaxInteger, int DefaultInteger);

    // The interface for the program: simply call get() and set()
    // as required. The wxConfigEntry and wxConfigEntryManager will
    // executed read() from the wxConfig as necessary (note that,
    // currently, modified entries are written only after calling 
    // writeAllConfigEntries())
    virtual void get (int & Integer);
    virtual int get ();
    virtual void get (unsigned int & UInt);
    virtual void get (long & Long);
    virtual void get (unsigned long & ULong);
    // NOTE set() does not check the range of the provided Integer -
    // (min, max) checking is done only for what is read from the wxConfig
    // (it is assumed that the program does not provide invalid values)
    virtual void set (const int & Integer);

    // overloaded read(), write(): called by wxConfigEntryManager
    // - for MessageString, see class notes
    virtual bool write (wxString & MessageString);
    virtual bool read (wxString & MessageString);
    virtual void setToDefault ();

  protected:
    // range of allowed integers:
    int m_MinInteger;
    int m_MaxInteger;
    // default value if key in the wxConfig is not present or invalid
    int m_DefaultInteger;

    int m_CurrentValue;
};

//-------------------------------------------------------------------------

class wxBooleanConfigEntry : public wxConfigEntry
{
  public:
    wxBooleanConfigEntry ();

    void init (const wxString & EntryPath, bool DefaultBoolean);
    void setDefault (bool DefaultBoolean);

    // The interface for the program: simply call get() and set()
    // as required. The wxConfigEntry and wxConfigEntryManager will
    // executed read() from the wxConfig as necessary (note that,
    // currently, modified entries are written only after calling 
    // writeAllConfigEntries())
    virtual void get (bool & Boolean);
    virtual bool get ();
    virtual void set (const bool & Boolean);

#if defined(__WXMSW__)
    // for convenience in MFC based programs:
    virtual void get (BOOL & Boolean);
    virtual void set (const BOOL & Boolean);
#endif

    // overloaded read(), write(): called by wxConfigEntryManager
    // - for MessageString, see class notes
    virtual bool write (wxString & MessageString);
    virtual bool read (wxString & MessageString);
    virtual void setToDefault ();

  protected:
    // default value if key in the wxConfig is not present or invalid
    bool m_DefaultBoolean;

    bool m_CurrentValue;
};

//-------------------------------------------------------------------------

class wxStringConfigEntry : public wxConfigEntry
{
  public:
    wxStringConfigEntry();

    void init (const wxString & EntryPath, const wxString & DefaultString);
    void setDefault (const wxString & DefaultString);

    // The interface for the program: simply call get() and set()
    // as required. The wxConfigEntry and wxConfigEntryManager will
    // executed read() from the wxConfig as necessary (note that,
    // currently, modified entries are written only after calling 
    // writeAllConfigEntries())
    virtual void get (wxString & String);
    virtual wxString get ();
    virtual void set (const wxString & String);

    // overloaded read(), write(): called by wxConfigEntryManager
    // - for MessageString, see class notes
    virtual bool write (wxString & MessageString);
    virtual bool read (wxString & MessageString);
    virtual void setToDefault ();

  protected:
    // default value if key in the wxConfig is not present or invalid
    wxString m_DefaultString;

    wxString m_CurrentValue;
};

//-------------------------------------------------------------------------

class wxDoubleConfigEntry : public wxConfigEntry
{
  public:
    wxDoubleConfigEntry();

    void init (const wxString & EntryPath,
               double MinDouble, double MaxDouble, double DefaultDouble);
    void setRangeAndDefault (double MinDouble, double MaxDouble, double DefaultDouble);

    // The interface for the program: simply call get() and set()
    // as required. The wxConfigEntry and wxConfigEntryManager will
    // executed read() from the wxConfig as necessary (note that,
    // currently, modified entries are written only after calling 
    // writeAllConfigEntries())
    virtual void get (double & Double);
    virtual double get ();
    // virtual void get (float & Float);
    // NOTE set() does not check the range of the provided Double -
    // (min, max) checking is done only for what is read from the wxConfig
    // (it is assumed that the program does not provide invalid values)
    virtual void set (const double & Double);

    // overloaded read(), write(): called by wxConfigEntryManager
    // - for MessageString, see class notes
    virtual bool write (wxString & MessageString);
    virtual bool read (wxString & MessageString);
    virtual void setToDefault ();

  protected:
    // range of allowed integers:
    double m_MinDouble;
    double m_MaxDouble;
    // default value if key in the wxConfig is not present or invalid
    double m_DefaultDouble;

    double m_CurrentValue;
};

//-------------------------------------------------------------------------

class wxDateTimeConfigEntry : public wxConfigEntry
{
  public:
    wxDateTimeConfigEntry();

    void init (const wxString & EntryPath, 
               const wxDateTime & MinDateTime, 
               const wxDateTime & MaxDateTime,
               const wxDateTime & DefaultDateTime);
    void setRangeAndDefault (const wxDateTime & MinDateTime, const wxDateTime & MaxDateTime, 
                             const wxDateTime & DefaultDateTime);

    // The interface for the program: simply call get() and set()
    // as required. The wxConfigEntry and wxConfigEntryManager will
    // executed read() from the wxConfig as necessary (note that,
    // currently, modified entries are written only after calling 
    // writeAllConfigEntries())
    virtual void get (wxDateTime & DateTime);
    virtual wxDateTime get ();
    virtual void set (const wxDateTime & DateTime);

    // overloaded read(), write(): called by wxConfigEntryManager
    // - for MessageString, see class notes
    virtual bool write (wxString & MessageString);
    virtual bool read (wxString & MessageString);
    virtual void setToDefault ();

    bool readDateTime(wxDateTime & DateTime);

  protected:
    // default value if key in the wxConfig is not present or invalid
    wxDateTime m_MinDateTime;
    wxDateTime m_MaxDateTime;
    wxDateTime m_DefaultDateTime;

    wxDateTime m_CurrentValue;
};

//-------------------------------------------------------------------------

#define M_IsUndefExportMagic 1
#    include "WxExtLibAliases.h"
#undef M_IsUndefExportMagic

//-------------------------------------------------------------------------

#endif
