
/*
-------------------------------------------------------------------------
This file is part of WxWidgetsExtensions library.
-------------------------------------------------------------------------

WxWidgetsExtensions library 0.7.1
-----------------------------

COPYRIGHT NOTICE:

WxWidgetsExtensions library Copyright (c) 2003, 2004 Daniel Kps.

The WxWidgetsExtensions library and associated documentation files (the
"Software") is provided "AS IS".  The author(s) disclaim all
warranties, expressed or implied, including, without limitation, the
warranties of merchantability and of fitness for any purpose.  The
author(s) assume no liability for direct, indirect, incidental,
special, exemplary, or consequential damages, which may result from
the use of or other dealings in the Software, even if advised of the
possibility of such damage.

Permission is hereby granted, free of charge, to any person obtaining
a copy of this Software, to deal in the Software without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Software,
and to permit persons to whom the Software is furnished to do so,
subject to the following conditions:

 1. The origin of this source code must not be misrepresented.
 2. Altered versions must be plainly marked as such and must not be
    misrepresented as being the original source.
 3. This Copyright notice may not be removed or altered from any 
    source or altered source distribution.

End of WxWidgetsExtensions library Copyright notice

-------------------------------------------------------------------------
*/

#if defined(__GNUG__) && (!defined(__APPLE__)) && (!(defined M_NoPragmaInterface))
    #pragma implementation "WxMisc.cpp"
#endif

// For compilers that support precompilation, includes "wx/wx.h".
#include "wx/wxprec.h"

#ifdef __BORLANDC__
    #pragma hdrstop
#endif

#include "WxMisc.h"

#include <wx/mstream.h>

#include <wx/dc.h>
#include <wx/dcclient.h>

#ifdef M_IsFilteredDIBEnabled
#   include "ImageMagickExtensions.h"
#endif

#include "safecast.h"

//=========================================================================

wxMinimalTimeInterval::wxMinimalTimeInterval ()
{
    m_IsInitialized = FALSE;
}
    
void wxMinimalTimeInterval::setMinimalTimeInterval (long MinimalTimeIntervalMS)
{
    m_MinimalTimeIntervalMS = MinimalTimeIntervalMS;
}

void wxMinimalTimeInterval::setBegin ()
{
#ifdef M_MinimalTimeInterval_UseGetTickCount
    m_BeginUpTimeMS = ::GetTickCount();
#else
    // m_StopWatch.Start ();
    m_BeginTimeMSLongLong = wxGetLocalTimeMillis();
#endif
    m_IsInitialized = TRUE;
}

bool wxMinimalTimeInterval::check ()
{
    // return TRUE if start of interval is unspecified, i.e. no start event
    // has taken place
    if (!m_IsInitialized)
      return TRUE;

#ifdef M_MinimalTimeInterval_UseGetTickCount
    DWORD CurrentUpTimeMS = ::GetTickCount ();
    DWORD ElapsedTimeMS;
    if (m_BeginUpTimeMS > CurrentUpTimeMS)
      {
        // handle timer overflow?
//      CurrentUpTimeMS += 0xFFFFFFFF - CurrentUpTimeMS;
//      m_BeginUpTimeMS = 0;
        ElapsedTimeMS = CurrentUpTimeMS - m_BeginUpTimeMS;
      }
    else
      {
        ElapsedTimeMS = CurrentUpTimeMS - m_BeginUpTimeMS;
      }
    
    if (ElapsedTimeMS >= m_MinimalTimeIntervalMS)
      {
        return TRUE;
      }
    else
      {
        return FALSE;
      }
#else
    // BUG doesn't handle timer overflows (will happen after 
    // about 48 days)
    // return (m_StopWatch.Time () > m_MinimalTimeIntervalMS);

    // alternative implementation: use wxGetLocalTimeMillis() which
    // uses wxLongLong and thus has no overflow for about eternal years
    wxLongLong CurrentTimeMSLongLong = wxGetLocalTimeMillis();
    wxLongLong MinimalTimeIntervalMSLongLong = m_MinimalTimeIntervalMS;
    wxLongLong ElapsedTimeMSLongLong = CurrentTimeMSLongLong - m_BeginTimeMSLongLong;

    // WARN don't know if ElapsedTimeMS<0 may happen - under e.g. Win9x this seems to be 
    // the case after adjusting the system clock backwards; however, adjusting
    // the system clock forwards will yield incorrect results too -
    // TODO check if adjusting system time influences running processes under 
    // Linux, Unix
    // WARN workaround for ElapsedTimeMS < 0
    if (ElapsedTimeMSLongLong < 0)
      {
        setBegin();
        return FALSE;
      }
    if (ElapsedTimeMSLongLong >= MinimalTimeIntervalMSLongLong)
      {
        return TRUE;
      }
    else
      {
        return FALSE;
      }
#endif
}

//=========================================================================

void convertAsciiToHtml (const char * AsciiStringCharPtr, wxString & HtmlString,
                         bool IsTranslateNewlines)
{
    const char * CurrentAsciiStringCharPtr = NULL;

    HtmlString.Clear();

    // determine length of resulting HTML string and pre-alloc to
    // this length to avoid multiple reallocations
    int HtmlStringLength = 0;
    for (CurrentAsciiStringCharPtr = AsciiStringCharPtr; 
         *CurrentAsciiStringCharPtr != '\0'; ++CurrentAsciiStringCharPtr)
      {
        int CharAsInt = *CurrentAsciiStringCharPtr;
        switch (CharAsInt)
          {
          case '<':
            HtmlStringLength += 4;
            break;
          case '>':
            HtmlStringLength += 4;
            break;
          case '&':
            HtmlStringLength += 5;
            break;
          case '"':
            HtmlStringLength += 6;
            break;
          case '\n':
            HtmlStringLength += (IsTranslateNewlines ? 4 : 1);
            break;
          default:
            HtmlStringLength += 1;
            break;
          }
      }

    HtmlString.Alloc (HtmlStringLength);

    for (CurrentAsciiStringCharPtr = AsciiStringCharPtr; 
         *CurrentAsciiStringCharPtr!='\0'; ++CurrentAsciiStringCharPtr)
      {
        int CharAsInt = *CurrentAsciiStringCharPtr;
        switch (CharAsInt)
          {
          case '<':
            HtmlString += "&lt;";
            break;
          case '>':
            HtmlString += "&gt;";
            break;
          case '&':
            HtmlString += "&amp;";
            break;
          case '"':
            HtmlString += "&quot;";
            break;
          case '\n':
            if (IsTranslateNewlines)
              HtmlString += "<br>";
            else
              HtmlString += (char) CharAsInt;
            break;
          default:
            HtmlString += (char) CharAsInt;
            break;
          }
      }
}

wxString convertAsciiToHtml (const char * AsciiStringCharPtr,
                             bool IsTranslateNewlines)
{
    wxString HtmlString;
    convertAsciiToHtml (AsciiStringCharPtr, HtmlString, IsTranslateNewlines);

    return HtmlString;
}

//-------------------------------------------------------------------------

wxString formatColourToHtmlParam (const wxColour & Colour)
{
    return wxString::Format ("#%02x%02x%02x", 
                             Colour.Red(), Colour.Green(), Colour.Blue());
}

//=========================================================================

bool getHtmlTagParamAsDouble (const wxHtmlTag & HtmlTag,
                              const wxString & ParamNameString, double * Double)
{
    wxString ParamValueString = HtmlTag.GetParam (ParamNameString);
    return parseDoubleFromAnyLocaleString (ParamValueString, Double);
}

bool parseDoubleFromAnyLocaleString (const wxString & InputValueString, double * Double)
{
    // HACK to get around problem with locale for float/double formatting
    // wxString::Format() and HtmlTag::ScanParam() seem to use current locale 
    // settings, but sometimes we want to parse (as strings) hard-coded double 
    // parameters (e.g. occuring in HTML templates)
    // Normally ScanParam() would fail (depending on the locale setting).

    // BUG
    // 1) the used approach probably fails if the input strings contains 
    //    thousands-separators (grouping characters)
    // 2) the used approach probably fails if a comma is used to separate
    //    a sequence of numbers (possibly written without spaces)

    // WARN
    // - don't know if there are other decimal separator charcaters than
    //   comma and point. If so, parsing for them would have to be added too.

    wxString ValueString (InputValueString);

    // try to parse in two different ways: once with occurences of
    // points replaced by commas and once with occurences of commas
    // replaced by points
    int WithCommaParsedCharacterCount = 0;
    double WithCommaParsedDouble = 0.;
    ValueString.Replace (".", ",", TRUE);
    int WithCommaParsedArgumentCount = wxSscanf (ValueString.c_str(), "%lf%n", 
                                                 & WithCommaParsedDouble,
                                                 & WithCommaParsedCharacterCount);

    int WithPointParsedCharacterCount = 0;
    double WithPointParsedDouble = 0.;
    ValueString.Replace (",", ".", TRUE);
    int WithPointParsedArgumentCount = wxSscanf (ValueString.c_str(), "%lf%n", 
                                                 & WithPointParsedDouble,
                                                 & WithPointParsedCharacterCount);

    // determine which of both possible results to use: if both ways we could
    // parse at least on parameter, we use the result where more characters were
    // parsed (quite hacky, admittedly). It would be better to ask for 
    // the setting of the locale, but i don't know which function is available
    // (if any) for this.
    bool IsUsePointParsedDouble = TRUE;
    if (WithPointParsedArgumentCount > 0
        && WithPointParsedArgumentCount == 0)
      {
        IsUsePointParsedDouble = TRUE;
      }
    else if (WithPointParsedArgumentCount == 0
        && WithCommaParsedArgumentCount > 0)
      {
        IsUsePointParsedDouble = FALSE;
      }
    else if (WithPointParsedArgumentCount > 0
        && WithCommaParsedArgumentCount > 0)
      {
        IsUsePointParsedDouble = (WithPointParsedCharacterCount >= WithCommaParsedCharacterCount);
      }

    // set function's output values
    int ParsedArgumentCount = IsUsePointParsedDouble 
      ? WithPointParsedArgumentCount : WithCommaParsedArgumentCount;
    if (IsUsePointParsedDouble)
      {
        * Double = WithPointParsedDouble;
      }
    else
      {
        * Double = WithCommaParsedDouble;
      }

    return ParsedArgumentCount > 0;
}

//=========================================================================

void getDecomposedDateFormatString (const wxString & InputFormatString,
                                    wxString & DecomposedDateFormatString)
{
    // our sample date: Monday, 22 October 1990
    // - the numbers should be unique (don't use 12 December or the like)
    // - the numbers should be 2-digit (so we don't have to care if a leading
    //   zero is printed or is not printed) - to simplify things a little
    wxDateTime SampleDateTime (/* day */ 22, 
                             /* month */ wxDateTime::Oct, 
                             /* year */ 1990);
    //  wxDateTime_t hour = 0, wxDateTime_t minute = 0,
    // wxDateTime_t second = 0, wxDateTime_t millisec = 0)    

    // WARN
    // - maybe it would be more appropriate to call CallStrftime() directly
    //   instead of calling wxDateTime::Format()
    // - should check if wxDateTime::Format()/CallStrftime() returned
    //   an error

    // get abbreviated and full weekday string for Monday for current locale
    wxString AbbreviatedSampleWeekdayString = SampleDateTime.Format ("%a"); 
    wxString FullSampleWeekdayString = SampleDateTime.Format ("%A");

    // get abbreviated and full month name for October for current locale
    wxString AbbreviatedSampleMonthString = SampleDateTime.Format ("%b"); 
    wxString FullSampleMonthString = SampleDateTime.Format ("%B");

    // format the sample date
    wxString SampleDateTimeString = SampleDateTime.Format (InputFormatString);

    // replace occurences of "%" with "%%" first:
    SampleDateTimeString.Replace ("%", "%%");

    // BUG: simply searching/replacing "%X" is incorrect if the format 
    // string contains the sequence "%%" (then, the "%a" in "%%a" is not
    // a placeholder) - to fix, add full parsing of format string
    // 
    SampleDateTimeString.Replace (FullSampleWeekdayString, "%A");
    SampleDateTimeString.Replace (AbbreviatedSampleWeekdayString, "%a");
    SampleDateTimeString.Replace ("22", "%d");
    SampleDateTimeString.Replace (FullSampleMonthString, "%B");
    SampleDateTimeString.Replace (AbbreviatedSampleMonthString, "%b");
    SampleDateTimeString.Replace ("10", "%m");
    SampleDateTimeString.Replace ("1990", "%Y");
    SampleDateTimeString.Replace ("90", "%y"); // maybe use %Y here too

    // now, SampleDateTimeString should contain a valid format string according
    // to the current locale setting
    DecomposedDateFormatString = SampleDateTimeString;

    // TODO: add checking if it was actually possible to determine
    // the decomposed format string; add a fall-back
}

void getDecomposedDateFormatStringX (wxString & DecomposedDateFormatString,
                                     bool IsLongFormat)
{
    if (IsLongFormat)
      {
#if defined(__WXMSW__)
#if (!defined(_MSC_VER)) && (!defined(__BORLANDC__))
#   pragma message ("Please check if your compiler/OS uses '%%#x' or '%%Ex' for alternative date format")
#endif
        // the format string to indicate alternative date/time 
        // representation differs in MSW/VC++, Borland C++
        // TODO check which other compilers use %#c instead of %Ec
        getDecomposedDateFormatString ("%#x", DecomposedDateFormatString);
#else
        getDecomposedDateFormatString ("%Ex", DecomposedDateFormatString);
#endif
      }
    else
      {
        getDecomposedDateFormatString ("%x", DecomposedDateFormatString);
      }
}

bool removeWeekdayFromDateFormatString (const wxString & DateFormatString,
                                        wxString & WithoutWeekdayDateFormatString)
{
    // make format string with weekday placeholder removed:
    // remove %a or %A, remove following spaces, punctuation characters etc.
    // WARN may have to remove spaces, punctuation characters before %a, %A as well
    //
    // BUG: simply searching for "%X" is incorrect if the format string
    // contains the sequence "%%" (then, the "%a" in "%%a" is not
    // a placeholder) - to fix, add full parsing of format string
    // 
    int WeekdayPlaceholderCharIndex = DateFormatString.Find ("%a");
    if (WeekdayPlaceholderCharIndex == wxNOT_FOUND)
      WeekdayPlaceholderCharIndex = DateFormatString.Find ("%A");
    if (WeekdayPlaceholderCharIndex != wxNOT_FOUND)
      {
        int CurrentCharIndex = WeekdayPlaceholderCharIndex + 2 /* strlen ("%a") */;
        for (; cast_is_smaller (CurrentCharIndex, DateFormatString.Length());
             ++CurrentCharIndex)
          {
            int CurrentChar = DateFormatString [CurrentCharIndex];
            if (!(wxIsspace (CurrentChar)
                  || (wxIspunct (CurrentChar) && CurrentChar != '%')))
              break;
          }
        WithoutWeekdayDateFormatString = DateFormatString;
        WithoutWeekdayDateFormatString.Remove (WeekdayPlaceholderCharIndex, 
                                               CurrentCharIndex-WeekdayPlaceholderCharIndex);

        return TRUE;
      }
    return FALSE;
}

//-------------------------------------------------------------------------

void formatDate (const wxDateTime & DateTime, wxString & String, 
                 bool IsLongFormat)
{
    // NOTE date formatting using "%#x" or "%Ex"
    // sometimes wxDateTime uses its own implementation which 
    // doesn't understand 'E' or '#'. The wxWindows implementation
    // is apparently used for dates before 2 Janary 1970
    // (an assertion failure is reported, and the output string
    // is set to the format string minus the percent character)
    // 
    // following hack is not required anymore, now there
    // is getDecomposedDateFormatString() which works better
    // 
    // if (TextCtrlContentString.IsSameAs ("#x")
    //  || TextCtrlContentString.IsSameAs ("Ex"))
    //   {
    //  // since we don't know if the current locale has 
    //  // DAY MONTH or MONTH DAY order, we output the date with the
    //  // month name to avoid any ambiguities
    //  TextCtrlContentString = TempDateTime.Format("%d %B %Y");
    // }

    wxString DateFormatString;
    getDecomposedDateFormatStringX (DateFormatString, IsLongFormat);
    
    String = DateTime.Format (DateFormatString);
}

//-------------------------------------------------------------------------

// parseDate()
// BUG
// - correct parsing of input (in case of occurences of "%%") would
//   be required instead of simply replacing %X, if any of the format
//   strings tried contains "%%"
//
bool parseDate (const wxString & String, wxDateTime & DateTime)
{
    // wxDateTime::ParseDate() works quite well unless the year in the string
    // is 2 digit and less than 31... Such a date string is generated
    // by Format("%#x") if no locale or some English locale was set 
    // (so the output is e.g. 02/19/04)
    // ParseDate() doesn't handle such a date.
    //
    // ParseFormat() may only be able to parse formats like dd/mm/yy or
    // mm/dd/yy, because possibly 'strptime' is not available. 
    // In this case, ParseFormat() cannot parse a date like '2 March 2002'.
    //
    // the following hack was used: The solution is to use try first 
    // ParseFormat() and if it fails try ParseDate(). 
    // However, this still doesn't work well because of the following problem:
    // 
    // possibly BUG in wxDateTime::ParseFormat():
    // for "%x", it uses "%d/%m/%y" as format if the time zone is e.g. European.
    // But the locale setting may well differ from the time zone setting.
    // (scope: wxWindows 2.4.1). 
    // Same problem may exist for ParseDate().
    //
    //!! A solution to this problem would be to make a dummy print of 
    //!! a fixed date like 10 November 2012 (2012-11-10) and then 
    //!! look in the output string if it was printed in day-month 
    //!! or month-day order
    //!! we could even try to construct a format string from the dummy
    //!! result
    // (the last idea is now implemented with getDecomposedDateFormatString())
    //
    // hack is obsolete now since we have getDecomposedDateFormatString()
    // which works well
    // --->
    // WARN see bug in ParseFormat() above
    // const char * EndCharPtr = TempDateTime.ParseFormat (BeginCharPtr, "%x");
    // bool IsOk = TRUE;
    // if ((EndCharPtr == NULL) || (EndCharPtr == BeginCharPtr) || (*EndCharPtr != 0))
    //   {
    //  // try conversion with ParseDate() if ParseFormat() failed
    //  EndCharPtr = TempDateTime.ParseDate (BeginCharPtr);
    //  if ((EndCharPtr == NULL) || (EndCharPtr == BeginCharPtr) || (*EndCharPtr != 0))
    //    {
    //      IsOk = FALSE;
    //    }
    //   }
    // <----

    const char * BeginCharPtr = String.c_str();
    const char * EndCharPtr = NULL;

    // try parsing the input as short date format:
    wxString ShortDateFormatString;
    getDecomposedDateFormatStringX (ShortDateFormatString, false);
    // note: under some platforms/locals, %x is equal to "%d.%m.%Y,
    // but we must parse with 2-digit year first because a 2-digit year
    // in the input may be interpreted as first-century year by
    // wxDateTime::ParseFormat().
    ShortDateFormatString.Replace ("%Y", "%y"); // make year 2-digit
    EndCharPtr = DateTime.ParseFormat (BeginCharPtr, ShortDateFormatString);
    if (!((EndCharPtr == NULL) || (EndCharPtr == BeginCharPtr) || (*EndCharPtr != 0)))
      {
        return TRUE;
      }

    // try to parse input as short format, but with 4-digit year now
    ShortDateFormatString.Replace ("%y", "%Y");
      {
        EndCharPtr = DateTime.ParseFormat (BeginCharPtr, ShortDateFormatString);
        if (!((EndCharPtr == NULL) || (EndCharPtr == BeginCharPtr) || (*EndCharPtr != 0)))
          {
            return TRUE;
          }
      }

    // try to parse input as long date format
    wxString DateFormatString;
    getDecomposedDateFormatStringX (DateFormatString, true);
    EndCharPtr = DateTime.ParseFormat (BeginCharPtr, DateFormatString);
    if (!((EndCharPtr == NULL) || (EndCharPtr == BeginCharPtr) || (*EndCharPtr != 0)))
      {
        return TRUE;
      }

    // try to parse with weekday removed from long format string:
    wxString WithoutWeekdayDateFormatString;
    if (removeWeekdayFromDateFormatString (DateFormatString, 
                                           WithoutWeekdayDateFormatString))
      {
        const char * EndCharPtr = DateTime.ParseFormat (BeginCharPtr, 
                                                        WithoutWeekdayDateFormatString);
        if (!((EndCharPtr == NULL) || (EndCharPtr == BeginCharPtr) || (*EndCharPtr != 0)))
          {
            return TRUE;
          }
      }

    // try to parse input date in ISO-8601 format (e.g. 2004-06-01)
    // WARN 
    // - Danish locale (and some others) may use %d-%m-%y as short date format, 
    //   so this may be ambiguous with ISO-8601 %Y-%m-%d
    // - at least, it would be wise not to use ISO-8601 with a two-digit year
    wxString ISO8601DateFormatString = "%Y-%m-%d";
    int DummyYear = 0;
    int ParsedYearDigits = 0;
    if ((wxSscanf (BeginCharPtr, "%d%n-", & DummyYear, & ParsedYearDigits) >= 1)
        && ParsedYearDigits == 4)
      {
        EndCharPtr = DateTime.ParseFormat (BeginCharPtr, ISO8601DateFormatString);
        if (!((EndCharPtr == NULL) || (EndCharPtr == BeginCharPtr) || (*EndCharPtr != 0)))
          {
            return TRUE;
          }
      }

    return FALSE;
}

//=========================================================================

wxFormat::wxFormat (int Integer)
{
    init ("%d", Integer);
}

wxFormat::wxFormat (const char * PrintfFormatString, int Integer)
{
    init (PrintfFormatString, Integer);
}

wxFormat::wxFormat (long Long)
{
    init ("%ld", Long);
}

wxFormat::wxFormat (const char * PrintfFormatString, long Long)
{
    init (PrintfFormatString, Long);
}

wxFormat::wxFormat (const double & Double)
{
    init ("%f", Double);
}

wxFormat::wxFormat (const char * PrintfFormatString, const double & Double)
{
    init (PrintfFormatString, Double);
}

wxFormat::wxFormat (const char * CharPtr)
{
    init ("%s", CharPtr);
}

wxFormat::wxFormat (const char * PrintfFormatString, const char * CharPtr)
{
    init (PrintfFormatString, CharPtr);
}

wxFormat::wxFormat (const wxString & String)
{
    init ("%s", String);
}

wxFormat::wxFormat (const char * PrintfFormatString, const wxString & String)
{
    init (PrintfFormatString, String);
}

//-------------------------------------------------------------------------

wxFormat::~wxFormat ()
{
}

//-------------------------------------------------------------------------

void wxFormat::init (const char * PrintfFormatString, int Integer)
{
    m_FormattedValueString.Printf (PrintfFormatString, Integer);
}

void wxFormat::init (const char * PrintfFormatString, long Long)
{
    m_FormattedValueString.Printf (PrintfFormatString, Long);
}

void wxFormat::init (const char * PrintfFormatString, const double & Double)
{
    m_FormattedValueString.Printf (PrintfFormatString, Double);
}

void wxFormat::init (const char * PrintfFormatString, const char * CharPtr)
{
    m_FormattedValueString.Printf (PrintfFormatString, CharPtr);
}

void wxFormat::init (const char * PrintfFormatString, const wxString & String)
{
    m_FormattedValueString.Printf (PrintfFormatString, String.c_str());
}

void wxFormat::appendTo (wxString & String) const
{
    String += m_FormattedValueString;
}

//=========================================================================

wxString wxFormatMessage (const char * FormatString)
{
    wxMessageFormatter MessageFormatter_ (FormatString);

    return MessageFormatter_.getRef();
}

wxString wxFormatMessage (const char * FormatString, const wxFormat & Param1Format)
{
    wxMessageFormatter MessageFormatter_ (FormatString);
    MessageFormatter_ % Param1Format;

    return MessageFormatter_.getRef();
}

wxString wxFormatMessage (const char * FormatString, 
                          const wxFormat & Param1Format,
                          const wxFormat & Param2Format)
{
    wxMessageFormatter MessageFormatter_ (FormatString);
    MessageFormatter_ 
      % Param1Format
      % Param2Format;

    return MessageFormatter_.getRef();
}

wxString wxFormatMessage (const char * FormatString, 
                          const wxFormat & Param1Format,
                          const wxFormat & Param2Format,
                          const wxFormat & Param3Format)
{
    wxMessageFormatter MessageFormatter_ (FormatString);
    MessageFormatter_ 
      % Param1Format
      % Param2Format
      % Param3Format;

    return MessageFormatter_.getRef();
}

wxString wxFormatMessage (const char * FormatString, 
                          const wxFormat & Param1Format,
                          const wxFormat & Param2Format,
                          const wxFormat & Param3Format,
                          const wxFormat & Param4Format)
{
    wxMessageFormatter MessageFormatter_ (FormatString);
    MessageFormatter_ 
      % Param1Format
      % Param2Format
      % Param3Format
      % Param4Format;

    return MessageFormatter_.getRef();
}

wxString wxFormatMessage (const char * FormatString, 
                          const wxFormat & Param1Format,
                          const wxFormat & Param2Format,
                          const wxFormat & Param3Format,
                          const wxFormat & Param4Format,
                          const wxFormat & Param5Format)
{
    wxMessageFormatter MessageFormatter_ (FormatString);
    MessageFormatter_ 
      % Param1Format
      % Param2Format
      % Param3Format
      % Param4Format
      % Param5Format;

    return MessageFormatter_.getRef();
}

//=========================================================================

wxMessageFormatter MessageFormatter (const char * FormatString)
{
    return wxMessageFormatter (FormatString);
}

//-------------------------------------------------------------------------

wxString & operator+= (wxString & String, wxMessageFormatter & MessageFormatter)
{
    MessageFormatter.appendTo (String);
    return String;
}

//-------------------------------------------------------------------------

wxMessageFormatter::wxMessageFormatter (const char * FormatString)
{
    m_FormatString = FormatString;
}

wxMessageFormatter::~wxMessageFormatter ()
{
    int FormatCount = m_FormatArray.GetCount();
    for (int FormatIndex=0; FormatIndex<FormatCount; ++FormatIndex)
      {
        delete m_FormatArray[FormatIndex];
        m_FormatArray[FormatIndex] = NULL;
      }
}

//-------------------------------------------------------------------------

wxMessageFormatter & wxMessageFormatter::operator% (int Integer)
{
    addFormat (new wxFormat (Integer));
    return *this;
};

wxMessageFormatter & wxMessageFormatter::operator% (long Long)
{
    addFormat (new wxFormat (Long));
    return *this;
};

wxMessageFormatter & wxMessageFormatter::operator% (const double & Double)
{
    addFormat (new wxFormat (Double));
    return *this;
};

wxMessageFormatter & wxMessageFormatter::operator% (const char * CharPtr)
{
    addFormat (new wxFormat (CharPtr));
    return *this;
};

wxMessageFormatter & wxMessageFormatter::operator% (const wxString & String)
{
    addFormat (new wxFormat (String));
    return *this;
};

wxMessageFormatter & wxMessageFormatter::operator% (const wxFormat & Format)
{
    addFormat (new wxFormat (Format));
    return *this;
};

wxMessageFormatter & wxMessageFormatter::operator% (wxFormat * FormatPtr)
{
    addFormat (FormatPtr);
    return *this;
};

//-------------------------------------------------------------------------

wxMessageFormatter & wxMessageFormatter::addFormat (wxFormat * FormatPtr)
{
    m_FormatArray.Add (FormatPtr);
    return *this;
};

//-------------------------------------------------------------------------

void wxMessageFormatter::format ()
{
    int FormatStringCharCount = m_FormatString.Length();
    bool IsPreviousCharPercentChar = FALSE;

    int FormatIndex = 0;

    for (int FormatStringCharIndex=0;
         FormatStringCharIndex < FormatStringCharCount;
         ++FormatStringCharIndex)
      {
        const char * FormatStringCharPtr = & (m_FormatString.c_str() [FormatStringCharIndex]);

        if (IsPreviousCharPercentChar)
          {
            if (*FormatStringCharPtr == '%')
              {
                m_OutputString += '%';
              }
            else if (isdigit (*FormatStringCharPtr)
                     || (*FormatStringCharPtr == '+')
                     || (*FormatStringCharPtr == '{'))
              {
                bool IsParsed = FALSE;
                if (*FormatStringCharPtr == '+')
                  {
                    // used syntax is "%+"
                    ++FormatIndex;
                    IsParsed = TRUE;
                  }
                else if (*FormatStringCharPtr == '{')
                  {
                    // used syntax is "%{N}" (e.g. "%{12}", "%{5}"
                    int ParsedCharacterCount = 0;
                    if (sscanf (FormatStringCharPtr + 1, "%d%n", 
                                & FormatIndex,
                                & ParsedCharacterCount) >= 1)
                      {
                        IsParsed = TRUE;

                        FormatStringCharIndex += ParsedCharacterCount;
                        if (FormatStringCharPtr [ParsedCharacterCount + 1] == '}')
                          {
                            ++FormatStringCharIndex;
                          }
                      }
                  }
                else if (isdigit (*FormatStringCharPtr))
                  {
                    // used syntax is "%N" (single digit, e.g. "%2")
                    FormatIndex = ((int) *FormatStringCharPtr) - '0';
                    IsParsed = TRUE;
                  }

                if (IsParsed)
                  {
                    if (FormatIndex > 0
                        && cast_is_equal_smaller (FormatIndex, m_FormatArray.GetCount()))
                      {
                        m_FormatArray[FormatIndex-1] -> appendTo (m_OutputString);
                      }
                    else
                      {
                        m_OutputString += "[??]";
                        // TODO: could add logging if argument is missing
                      }
                  }
              }
            IsPreviousCharPercentChar = FALSE;
          }
        else
          {
            if (*FormatStringCharPtr != '%')
              m_OutputString += *FormatStringCharPtr;

            IsPreviousCharPercentChar = *FormatStringCharPtr == '%';
          }
      }
}

//-------------------------------------------------------------------------

const wxString & wxMessageFormatter::getRef ()
{
    format ();
    return m_OutputString;
}

wxMessageFormatter::operator const wxString & ()
{
    format ();
    return m_OutputString;
}

wxMessageFormatter::operator const char * ()
{
    format ();
    return m_OutputString;
}

//-------------------------------------------------------------------------

void wxMessageFormatter::appendTo (wxString & String)
{
    String += operator const wxString & ();
}

void wxMessageFormatter::assignTo (wxString & String)
{
    String = operator const wxString & ();
}

//=========================================================================

void fitInside (const wxRect & FrameRect, const wxSize & ObjectSize, 
                wxRect & FittedObjectRect, int AlignmentFlags)
{
    int FrameRectWidth = FrameRect.GetWidth();
    int FrameRectHeight = FrameRect.GetHeight();
    int FittedX = FrameRect.GetX();
    int FittedY = FrameRect.GetY();
    int FittedWidth;
    int FittedHeight;
    double FrameRectWidthHeightRatio = (double) FrameRectWidth
      / (double) FrameRectHeight;
    double ObjectWidthHeightRatio = (double) ObjectSize.GetWidth()
      / (double) ObjectSize.GetHeight();

    if (FrameRectWidthHeightRatio > ObjectWidthHeightRatio)
      {
        cast_assign (FittedWidth, FrameRectHeight * ObjectWidthHeightRatio);
        FittedHeight = FrameRectHeight;

        if (AlignmentFlags & wxALIGN_CENTER_HORIZONTAL)
          {
            FittedX += (FrameRectWidth - FittedWidth) / 2;
          }
        else if (AlignmentFlags & wxALIGN_RIGHT)
          {
            FittedX += (FrameRectWidth - FittedWidth);
          }
      }
    else
      {
        FittedWidth = FrameRectWidth;
        cast_assign (FittedHeight, FrameRectWidth * 1/ObjectWidthHeightRatio);

        if (AlignmentFlags & wxALIGN_CENTER_VERTICAL)
          {
            FittedY += (FrameRectHeight - FittedHeight) / 2;
          }
        else if (AlignmentFlags & wxALIGN_BOTTOM)
          {
            FittedY += (FrameRectHeight - FittedHeight);
          }
      }

    FittedObjectRect = wxRect (FittedX, FittedY, FittedWidth, FittedHeight);
}

//=========================================================================

#if defined(__WXMSW__)

// // wxGetInstance() is somehow needed for loadDataResource():
// // (as declared in ./include/wx/msw/private.h):
// WXDLLEXPORT HINSTANCE wxGetInstance();
// NOTE we use GetModuleHandle(NULL) instead of wxGetInstance()
// which may not work under Win16

// NOTE semantics not well defined for Win16: there, some function like
// releaseDataResource() or copying the data block would be needed
bool loadDataResource (const wxString & ResourceNameString, 
                       const char * ResourceTypeString,
                       unsigned char ** DataPtr, size_t * DataSize)
{
    bool IsCompleted = FALSE;

    for_once
      {
        HRSRC ResourceHandle = ::FindResource (::GetModuleHandle(NULL) /*wxGetInstance()*/, 
                                               ResourceNameString, ResourceTypeString);
        if (ResourceHandle == 0)
          break;

        HGLOBAL DataHandle = NULL;
        if (DataPtr != NULL)
          {
            // load and lock data block (if desired, i.e. if DataPtr != NULL)
            DataHandle = ::LoadResource (::GetModuleHandle(NULL) /*wxGetInstance()*/, 
                                         ResourceHandle);
            if (DataHandle == 0)
              break;

            *DataPtr = (unsigned char *) ::LockResource (DataHandle);
            if (*DataPtr == NULL)
              break;
          }
            
        if (DataSize != NULL)
          {
            // determine size of the data block (if desired)
            *DataSize = ::SizeofResource (GetModuleHandle (NULL) /*wxGetInstance()*/, 
                                          ResourceHandle);
          }

        // BUG for Win16:
        // UnlockResource() must be called for Win16, however; we must
        // probably do this after working with the data

        IsCompleted = TRUE;
      }
    
    return IsCompleted;
}

// void releaseDataResource ()
// {
//      if (DataPtr != NULL)
//        {
//          // UnlockResource() not needed for WIN32
// #ifndef __WIN32__
//          UnlockResource(DataHandle);
// #endif
//        }
    
//      // needed for Win16?
//      //  GlobalFree(hData);
// }

#endif // defined(__WXMSW__)

//=========================================================================

bool loadImageFromMemory (const unsigned char * ImageData, size_t ImageDataSize,
                          wxImage & Image)
{
    wxMemoryInputStream MemoryInputStream (ImageData, ImageDataSize);
    return Image.LoadFile (MemoryInputStream);
}

bool loadBitmapFromMemory (const unsigned char * ImageData, size_t ImageDataSize,
                           wxBitmap ** Bitmap)
{
    *Bitmap = NULL;

    bool IsOk = FALSE;
    for_once
      {
        wxImage Image;
        if (! loadImageFromMemory (ImageData, ImageDataSize, Image))
          break;

        *Bitmap = new wxBitmap (Image);
        if (*Bitmap == NULL)
          break;

        IsOk = TRUE;
      }

    return IsOk;
}

#if defined(__WXMSW__)

bool loadImageFromResource (const wxString & ImageDataResourceNameString, 
                            wxImage & Image)
{
    unsigned char * DataPtr = NULL;
    size_t DataSize = 0;
    if (!loadDataResource (ImageDataResourceNameString, RT_RCDATA,
                           & DataPtr, & DataSize))
      return FALSE;

    return loadImageFromMemory (DataPtr, DataSize, Image);
}

bool loadBitmapFromResource (const wxString & ImageDataResourceNameString, 
                             wxBitmap ** Bitmap)
{
    bool IsOk = FALSE;

    *Bitmap = NULL;

    for_once
      {
        unsigned char * DataPtr = NULL;
        size_t DataSize = 0;
        if (!loadDataResource (ImageDataResourceNameString, RT_RCDATA,
                               & DataPtr, & DataSize))
          break;
    
        if (!loadBitmapFromMemory (DataPtr, DataSize, Bitmap))
          break;

        IsOk = TRUE;
      }

    if (!IsOk)
      *Bitmap = new wxBitmap ();

    return IsOk;
}

#endif // defined(__WXMSW__)

//=========================================================================

// doTextLineWrap()
// - text extents are calculated for the font currently set for the DC
// Possible improvements:
// - Insert e.g. '\r\n' or '\r\r\n' when having a 'soft' line break (not forced)
//   instead of just '\n'. This way, the soft line breaks could be un-done
//   again (because then they would be distinguishable from the hard line breaks).
//   This could be useful if a control want's to re-do line wrapping 
//   e.g. after a size change notification.
void doTextLineWrap (const wxString & InputString, 
                     int MaxLineWidth,
                     wxDC & DC,
                     wxString & ResultString)
{
    wxString CurrentLineString;
    // pre-allocate line-string to improve performance:
    CurrentLineString.Alloc (512);

    const char * CurrentCharPtr = InputString.c_str();
    // const char * LineStartCharPtr = CurrentCharPtr;
    const char * LastBreakCharPtr = NULL;

    int CurrentLineCharCount = 0;
    int LastBreakCharCount = 0;

    bool IsJustAfterWrap;
    bool IsAfterWrap = FALSE;
    while (1)
      {
        IsJustAfterWrap = FALSE;
        bool IsBreakAllowed = FALSE;
        bool IsBreakForced = FALSE;

        int CurrentChar = *CurrentCharPtr;

        if (IsAfterWrap)
          {
            if (!(CurrentChar == '\0'
                  || wxIsspace (CurrentChar))
                || CurrentChar == '\n')
              IsAfterWrap = FALSE;
          }

        if (!IsAfterWrap)
          {
            if (CurrentChar == '\0'
                || wxIsspace (CurrentChar))
              IsBreakAllowed = TRUE;

            if (CurrentChar == '\n'
                || CurrentChar == '\0')
              IsBreakForced = TRUE;
          }

        if (IsBreakAllowed
            || IsBreakForced)
          {
            int Width = 0;
            int Height = 0;
            // NOTE there is also wxWindow::GetTextExtent, maybe 
            // one could optionally use this function instead of wxDC::GetTextExtent
            DC.GetTextExtent (CurrentLineString, & Width, & Height);
        
            bool IsOverfull = (Width > MaxLineWidth);

            if (IsOverfull
                || IsBreakForced)
              {
                const char * ActualBreakCharPtr = NULL;
                if (IsOverfull)
                  {
                    ActualBreakCharPtr = LastBreakCharPtr;
                    if (ActualBreakCharPtr == NULL)
                      ActualBreakCharPtr = CurrentCharPtr;
                  }
                else
                  {
                    ActualBreakCharPtr = CurrentCharPtr;
                  }

                if (IsOverfull
                    && LastBreakCharPtr != NULL)
                  {
                    CurrentLineString.Truncate (LastBreakCharCount);
                  }

                ResultString.Append (CurrentLineString);
                ResultString.Append ("\n");
                
                // set string to empty but retain allocated memory:
                CurrentLineString.Empty ();

                // reposition pointers
                // LineStartCharPtr = ActualBreakCharPtr;
                CurrentCharPtr = ActualBreakCharPtr;
                CurrentChar = *CurrentCharPtr;
                LastBreakCharPtr = NULL;
                LastBreakCharCount = 0;
                CurrentLineCharCount = 0;

                if (CurrentChar == '\n')
                  {
                    ++CurrentCharPtr;
                    CurrentChar = *CurrentCharPtr;
                  }

                IsAfterWrap = TRUE;
                IsJustAfterWrap = TRUE;
              }
            else
              {
                LastBreakCharPtr = CurrentCharPtr;
                LastBreakCharCount = CurrentLineCharCount;
              }
          }

        if (CurrentChar == '\0')
          break;

        if (!IsAfterWrap)
          {
            CurrentLineString.Append ((char) CurrentChar);
            ++CurrentLineCharCount;
          }

        if (!IsJustAfterWrap)
          {
            ++CurrentCharPtr;
          }
      }
}

//-------------------------------------------------------------------------

void doTextLineWrap (const wxString & InputString, 
                     int MaxLineWidth,
                     wxWindow * Window,
                     wxString & ResultString)
{
   wxClientDC ClientDC (Window);
   wxFont & Font = Window -> GetFont ();

   ClientDC.SetFont (Font);
   doTextLineWrap (InputString, MaxLineWidth, ClientDC, ResultString);

   // WARN don't know if resetting old font is required
}

//=========================================================================

wxMultiLineText::wxMultiLineText ()
{
    m_TextCtrl = NULL;
    m_HtmlWindow = NULL;
    m_StaticText = NULL;

    m_MultiLineTextStyle = 0;
}

void wxMultiLineText::create (wxWindow * ParentWindow,
                              int WindowIdent,
                              int CtrlStyle,
                              const wxSize & MinSize,
                              int MultiLineTextStyle)
{
    wxSize UsedSize = MinSize;

    m_MultiLineTextStyle = MultiLineTextStyle;

    // determine what kind of borders to use:
    if (! (CtrlStyle & wxBORDER_NONE)
        && !(CtrlStyle & wxBORDER_SIMPLE))
      {
        // default is wxSUNKEN_BORDER:
        CtrlStyle |= wxSUNKEN_BORDER;
      }

    // determine if wxTextCtrl or wxHtmlWindow should be used:
    bool IsCreateTextCtrl = TRUE;

#if ((defined __WXMOTIF__) || defined(__WXGTK__))
    // with Lesstif/Motif?, we don't want to use wxTextCtrl 
    // because it uses a fixed-size typewriter-like font 
    // and setting colours is not possible
    //
    // GTK (1.2.7) wxTextCtrl ignores setting foreground, background 
    // colours or not using sunken border
    // 
    // default for Motif/Lesstif, GTK: use wxHtmlWindow
    IsCreateTextCtrl = FALSE;
#endif

    if (MultiLineTextStyle & wxFORCE_HTMLWINDOW)
      IsCreateTextCtrl = FALSE;

    if (MultiLineTextStyle & wxFORCE_TEXTCTRL)
      IsCreateTextCtrl = TRUE;

    if (MultiLineTextStyle & wxHTML_MODE)
      IsCreateTextCtrl = FALSE;

    // create the desired control type
    if (IsCreateTextCtrl)
      {
        // borders: wxTextCtrl seems to normally have a border.
        // No border can be requested with wxBORDER_NONE.
        
        long TextCtrlStyle = wxTE_MULTILINE | wxTE_LINEWRAP | wxTE_READONLY;
        /* | wxVSCROLL | wxTE_RICH | wxBORDER_NONE */
        // if (! (CtrlStyle & wxSUNKEN_BORDER))
        //  TextCtrlStyle |= wxBORDER_NONE;
        TextCtrlStyle |= CtrlStyle;
        
        m_TextCtrl = new wxTextCtrl (ParentWindow, WindowIdent, _T(""),
                                     wxDefaultPosition, UsedSize,
                                     TextCtrlStyle,
                                     wxDefaultValidator);

        m_TextCtrl -> SetBackgroundColour (ParentWindow -> GetBackgroundColour());
      }
    else
      {
        // borders: wxHtmlWindow seems to support
        // wxSIMPLE_BORDER, wxSUNKEN_BORDER, (no border if no flag specified),
        // maybe others
        m_HtmlWindow = new wxHtmlWindow (ParentWindow, WindowIdent, 
                                         wxDefaultPosition, 
                                         UsedSize,
                                         CtrlStyle);
        m_HtmlWindow -> SetBorders (2);
        m_HtmlWindow -> SetBackgroundColour (ParentWindow -> GetBackgroundColour());
      }

//  // wxStaticText apparently ignores wxVSCROLL (at least under
//  // this X11 and this Lesstif), so it isn't very useful for our cause
//  // of displaying, possibly very long, multi-line text
//     m_StaticText = new wxStaticText (ParentWindow, WindowIdent, _T(""),
//                                   wxDefaultPosition, UsedSize,
//                                   /* WindowStyle */ wxVSCROLL);
    
}

//-------------------------------------------------------------------------

void wxMultiLineText::setText (const wxString & String)
{
    if (m_TextCtrl != NULL)
      {
#if defined(__WXMSW__)
        // the RichEdit supports line wrapping
        m_TextCtrl -> SetValue (String);
#else
        // under X11, the wxTextCtrl doesn't support wrapping of long
        // lines by itself, here we call doTextLineWrap()
        // (note: from wxWindows 2.6 (2.5.x) on, the wxX11/wxUniv wxTextCtrl 
        // will have support for line wrap)
        //
        // GTK 1.x: the wxTextCtrl supports wxTE_LINEWRAP, but
        // wxTextCtrl::GetNumberOfLines() only counts newlines
        // (but we are interessted in the number of lines after 
        // doing the wrapping) - because of this problem, we use
        // doTextLineWrap() for GTK as well
        // 
        wxSize ClientSize = m_TextCtrl -> GetClientSize();
        wxString ResultString;

        int MaxLineWidth = ClientSize.GetWidth();
        // HACK quick hack to account for possibly existing 
        // scrollbars:
        if (MaxLineWidth >= 25)
          MaxLineWidth -= 25;

#if defined(__WXMOTIF__)
        // the Lesstif wxTextCtrl has lot's of bordering around the
        // space actually usable for text, so subtract a little from
        // the horizontal size
        MaxLineWidth -= 20;
#endif

        // NOTE: the wxUniversal wxTextCtrl now (version 2.5.2 tested) has support 
        // for wrapping long lines, so calling doTextLineWrap() may not be required 
        // for this newer wxWindows versions
        // Also, this new wrapping mechanism has support for partial re-wrapping,
        // which is not available with doTextLineWrap().
        // (however, must check semantics of GetNumberOfLines())
        doTextLineWrap (String, MaxLineWidth, 
                        m_TextCtrl,
                        ResultString);
        m_TextCtrl -> SetValue (ResultString);
#endif
      }
    else if (m_HtmlWindow != NULL)
      {
        // WARN should add charset name in HTML <head> for currently used charset
        wxString HtmlString = "<html><body bgcolor=\"" 
          + formatColourToHtmlParam (m_HtmlWindow -> GetBackgroundColour())
          + "\" text=\""
          + formatColourToHtmlParam (m_HtmlWindow -> GetForegroundColour())
          + "\">\n"
          + "";
        if (m_MultiLineTextStyle & wxHTML_MODE)
          {
            HtmlString += String;
          }
        else
          {
            HtmlString += convertAsciiToHtml (String, TRUE);
          }
        HtmlString += "</body></html>";
        m_HtmlWindow -> SetPage (HtmlString);
      }
    else if (m_StaticText != NULL)
      {
        m_StaticText -> SetLabel (String);
      }
}

//-------------------------------------------------------------------------

wxWindow * wxMultiLineText::getWindow ()
{
   if (m_TextCtrl != NULL)
     return m_TextCtrl;
   else if (m_HtmlWindow != NULL)
     return m_HtmlWindow;
   else if (m_StaticText != NULL)
     return m_StaticText;
   else
     return NULL;
}

wxTextCtrl * wxMultiLineText::getTextCtrl ()
{
    return m_TextCtrl;
}

wxHtmlWindow * wxMultiLineText::getHtmlWindow ()
{
    return m_HtmlWindow;
}

wxStaticText * wxMultiLineText::getStaticText ()
{
    return m_StaticText;
}

//=========================================================================

IMPLEMENT_DYNAMIC_CLASS(wxColourBarWindow, wxWindow)

BEGIN_EVENT_TABLE (wxColourBarWindow, wxWindow)
    EVT_ERASE_BACKGROUND (wxColourBarWindow::OnEraseBackground)
    EVT_PAINT(wxColourBarWindow::OnPaint)
END_EVENT_TABLE()

//-------------------------------------------------------------------------

wxColourBarWindow::wxColourBarWindow ()
{
    m_Brush = *wxBLUE_BRUSH;
}

wxColourBarWindow::wxColourBarWindow (wxWindow * ParentWindow, int ControlIdent,
                                      const wxColour & Colour,
                                      const wxPoint & Position, 
                                      const wxSize & Size,
                                      long Style, const wxString& Name)
  : wxWindow (ParentWindow, ControlIdent, 
              Position, Size,
              Style, Name)
{
    m_Brush.SetColour (Colour);
    m_Brush.SetStyle (wxSOLID);
}

//-------------------------------------------------------------------------

void wxColourBarWindow::setBrush (const wxBrush & Brush)
{
    m_Brush = Brush;
}

//-------------------------------------------------------------------------

void wxColourBarWindow::OnEraseBackground (wxEraseEvent & WXUNUSED(EraseEvent))
{
    // wxWindow::OnEraseBackground (EraseEvent);
}

void wxColourBarWindow::OnPaint (wxPaintEvent & WXUNUSED(PaintEvent))
{
    wxPaintDC PaintWxDC (this);

    wxRect WindowRect = GetClientRect();

    PaintWxDC.SetPen (*wxTRANSPARENT_PEN);
    PaintWxDC.SetBrush (m_Brush);
    PaintWxDC.DrawRectangle (WindowRect);
}

//=========================================================================

IMPLEMENT_DYNAMIC_CLASS(wxItemWindow, wxWindow)

BEGIN_EVENT_TABLE (wxItemWindow, wxWindow)
    EVT_ERASE_BACKGROUND (wxItemWindow::OnEraseBackground)
    EVT_PAINT(wxItemWindow::OnPaint)
END_EVENT_TABLE()

//-------------------------------------------------------------------------

wxItemWindow::wxItemWindow ()
{
    m_IsSelectable = FALSE;
    m_IsFocusable = FALSE;
    m_StatusMessageTarget = NULL;

    init ();
}

wxItemWindow::wxItemWindow (wxWindow * ParentWindow, int ControlIdent,
                                    const wxPoint & Position, 
                                    const wxSize & Size,
                                    bool IsSelectable, bool IsFocusable,
                                    wxStatusMessageTarget * StatusMessageTarget,
                                    long Style, const wxString& Name)
  : wxWindow (ParentWindow, ControlIdent, 
              Position, Size,
              Style, Name)
{
    m_StatusMessageTarget = StatusMessageTarget;
    m_IsSelectable = IsSelectable;
    m_IsFocusable = IsFocusable;

    init ();
}

void wxItemWindow::init ()
{
    m_IsSelected = FALSE;
    m_IsFocused = FALSE;
    m_IsBorder = FALSE;
    m_IsUsingFocusedBackgroundBrush = FALSE;
}

//-------------------------------------------------------------------------

void wxItemWindow::setFocusedBackgroundBrush (const wxBrush & Brush)
{
    m_IsUsingFocusedBackgroundBrush = TRUE;
    m_FocusedBackgroundBrush = Brush;
}

//-------------------------------------------------------------------------

void wxItemWindow::OnEraseBackground (wxEraseEvent & WXUNUSED(EraseEvent))
{
    // wxWindow::OnEraseBackground (EraseEvent);
}

void wxItemWindow::OnPaint (wxPaintEvent & WXUNUSED(PaintEvent))
{
    // wxRect WindowRect = GetRect ();
    // wxSize ClientSize = GetClientSize();

    wxPaintDC PaintWxDC (this);

    wxRect WindowRect = GetClientRect();

    // NOTE BorderWidth must be <= FocusBorderWidth (if m_IsFocusable)
    // or <= SelectionBorderWidth (if m_IsSelectable && !m_IsFocusable)
    enum { FocusBorderWidth = 3, SelectionBorderWidth = 3, BorderWidth = 1 };

    // calculate rectangles for borders and item
    wxRect ItemRect = WindowRect;
    wxRect SelectionIndicationRect = WindowRect;
    if (m_IsBorder)
      {
        if (!m_IsFocusable && !m_IsSelectable)
          {
            ItemRect.Inflate (-BorderWidth -1, -BorderWidth -1);
            SelectionIndicationRect.Inflate (-BorderWidth -1, -BorderWidth -1);
          }
      }

    if (m_IsFocusable)
      {
        ItemRect.Inflate (-FocusBorderWidth -1, -FocusBorderWidth -1);
        // ItemRect.Offset (FocusBorderWidth, FocusBorderWidth);
        SelectionIndicationRect.Inflate (-FocusBorderWidth -1, -FocusBorderWidth -1);
        // SelectionIndicationRect.Offset (FocusBorderWidth, FocusBorderWidth);
      }

    if (m_IsSelectable)
      {
        ItemRect.Inflate (-SelectionBorderWidth, -SelectionBorderWidth);
        // ItemRect.Offset (SelectionBorderWidth, SelectionBorderWidth);
      }

    wxBrush NormalBackgroundBrush = wxBrush (GetBackgroundColour(), wxSOLID);
    const wxBrush * BackgroundBrush = NULL;
    if (m_IsFocusable 
        && m_IsFocused
        && m_IsUsingFocusedBackgroundBrush)
      {
        BackgroundBrush = & m_FocusedBackgroundBrush;
      }
    else
      {
        BackgroundBrush = & NormalBackgroundBrush;
      }

    // draw background "behind" borders and rectangles (if there are any 
    // rectangles to be drawn)
    // NOTE this creates a minimal flicker if rectangles are drawn on the
    // rectangle drawn here
    if (WindowRect != ItemRect)
      {
        wxRegion BackgroundRegion (WindowRect);
        BackgroundRegion.Subtract (ItemRect);
        wxRegionIterator BackgroundRegionIterator (BackgroundRegion);
        PaintWxDC.SetBrush (* BackgroundBrush);
        PaintWxDC.SetPen (* wxTRANSPARENT_PEN);
        while (BackgroundRegionIterator.HaveRects ())
          {
            PaintWxDC.DrawRectangle (BackgroundRegionIterator.GetRect());
            ++BackgroundRegionIterator;
          }
      }
    
    // draw borders and rectangles around item
    if (m_IsBorder)
      {
        if ((!m_IsFocusable && !m_IsSelectable)
            || (!m_IsFocusable && m_IsSelectable && !m_IsSelected)
            || (m_IsFocusable && !m_IsFocused))
          {
            // draw gray rectangle to indicate area
            PaintWxDC.SetBrush (* wxTRANSPARENT_BRUSH);
            PaintWxDC.SetPen (wxPen (wxColour (0xD0, 0xD0, 0xD0), BorderWidth, wxSOLID));
            PaintWxDC.DrawRectangle (WindowRect.GetX(), 
                                     WindowRect.GetY(),
                                     WindowRect.GetWidth(), 
                                     WindowRect.GetHeight());
          }
      }

    if (m_IsFocusable)
      {
        if (m_IsFocused)
          {
            // draw blue rectangle around image to indicate selection:
            // and draw background with a very light gray
            // PaintWxDC.SetBrush (wxBrush (wxColour (0xF0, 0xF0, 0xF0), wxSOLID));
            if (!m_IsUsingFocusedBackgroundBrush)
              {
                PaintWxDC.SetBrush (* wxTRANSPARENT_BRUSH);
              }
            else
              {
                PaintWxDC.SetBrush (* wxTRANSPARENT_BRUSH);
                // PaintWxDC.SetBrush (m_FocusedBackgroundBrush);
              }
            PaintWxDC.SetPen (wxPen (wxColour (192, 32, 32), FocusBorderWidth, wxSOLID));
            PaintWxDC.DrawRectangle (WindowRect.GetX(), 
                                     WindowRect.GetY(),
                                     WindowRect.GetWidth(), 
                                     WindowRect.GetHeight());
          }
      }

    if (m_IsSelectable)
      {
        if (m_IsSelected)
          {
            // draw blue rectangle around image to indicate selection:
            PaintWxDC.SetBrush (* wxTRANSPARENT_BRUSH);
            PaintWxDC.SetPen (wxPen (wxColour (32, 0, 160), SelectionBorderWidth, wxSOLID));
            PaintWxDC.DrawRectangle (SelectionIndicationRect.GetX(), 
                                     SelectionIndicationRect.GetY(),
                                     SelectionIndicationRect.GetWidth(), 
                                     SelectionIndicationRect.GetHeight());
          }
      }

    // draw item
    handleItemPaint (PaintWxDC, ItemRect, BackgroundBrush);
}

void wxItemWindow::handleItemPaint (wxPaintDC & PaintWxDC,
                                    const wxRect & ItemRect,
                                    const wxBrush * BackgroundBrush)
{
    PaintWxDC.SetBrush (* BackgroundBrush);
    PaintWxDC.SetPen (* wxTRANSPARENT_PEN);
    PaintWxDC.DrawRectangle (ItemRect);
}

//=========================================================================

#if defined(__WXMSW__)

void drawDIBFitted (wxDC & WxDC, int X, int Y, int Width, int Height,
                    WXHDIB DIBHandle,
                    const wxBrush * BackgroundBrush)
{
    for_once
      {
        if (DIBHandle == NULL)
          break;

        // get DIB width and height:
        BITMAPINFO * BitmapInfo = (LPBITMAPINFO) GlobalLock (DIBHandle);
        if (BitmapInfo == NULL)
          break;

        BITMAPINFOHEADER * BitmapInfoHeader = (LPBITMAPINFOHEADER) BitmapInfo;

        int ImageWidth = BitmapInfoHeader -> biWidth;
        int ImageHeight = BitmapInfoHeader -> biHeight;
        int ColorPaletteSize = getActualUsedColorCount (BitmapInfo) * sizeof (RGBQUAD);

        unsigned char * BitmapData = ((unsigned char *) BitmapInfo)
          + BitmapInfoHeader -> biSize
          + ColorPaletteSize;

        wxRect Rect (X, Y, Width, Height);
        wxSize Size (ImageWidth, ImageHeight);
        wxRect FittedRect;
        fitInside (Rect, Size, FittedRect, wxALIGN_CENTER_HORIZONTAL | wxALIGN_CENTER_VERTICAL);

        FittedRect.Inflate (-1);

        // draw background around image if desired
        if (BackgroundBrush != NULL)
          {
            wxRegion BackgroundRegion (X, Y, Width, Height);
            BackgroundRegion.Subtract (FittedRect);
            wxRegionIterator BackgroundRegionIterator (BackgroundRegion);
            while (BackgroundRegionIterator.HaveRects ())
              {
                WxDC.SetBrush (* BackgroundBrush);
                WxDC.SetPen (* wxTRANSPARENT_PEN);
                WxDC.DrawRectangle (BackgroundRegionIterator.GetRect());
                ++BackgroundRegionIterator;
              }
          }
    
        // draw DIB
        HDC DCHandle = (HDC) WxDC.GetHDC();
        ::SetStretchBltMode (DCHandle, COLORONCOLOR);
        ::StretchDIBits (DCHandle,
                         FittedRect.GetX(), FittedRect.GetY(), 
                         FittedRect.GetWidth(), FittedRect.GetHeight(),
                         0, 0, ImageWidth, ImageHeight,
                         (void *) BitmapData,
                         BitmapInfo,
                         DIB_RGB_COLORS, SRCCOPY);

        FittedRect.Inflate (1);

        WxDC.SetBrush (*wxTRANSPARENT_BRUSH);
        WxDC.SetPen (*wxBLACK_PEN);
        WxDC.DrawRectangle (FittedRect.GetX(), FittedRect.GetY(),
                            FittedRect.GetWidth(), FittedRect.GetHeight());

        GlobalUnlock (DIBHandle);
    }
}

#endif

//=========================================================================

#if defined(__WXMSW__)

wxImageProxy::wxImageProxy ()
{
    m_DIBFromFile = NULL;
}

wxImageProxy::~wxImageProxy ()
{
    freeDIB ();
}

void wxImageProxy::freeDIB ()
{
    if (m_DIBFromFile != NULL)
      {
        GlobalFree (m_DIBFromFile);
        m_DIBFromFile = NULL;
      }
}

void wxImageProxy::discard ()
{
    freeDIB ();
}

void wxImageProxy::draw (wxDC & PaintWxDC, int X, int Y, int Width, int Height,
                         EResolutionIdent ResolutionIdent,
                         const wxBrush * BackgroundBrush)
{
    for_once
      {
        WXHDIB DIBHandle = NULL;
        getDIB (ResolutionIdent, DIBHandle);
        if (DIBHandle == NULL)
          break;

        drawDIBFitted (PaintWxDC, X, Y, Width, Height,
                       DIBHandle,
                       BackgroundBrush);
      }
}

#endif // defined(__WXMSW__)

//=========================================================================

#if defined(__WXMSW__)

wxBitmapPaintHelper::wxBitmapPaintHelper (const wxString & ResourceIdentString)
{
    loadBitmap (ResourceIdentString);
};

wxBitmapPaintHelper::wxBitmapPaintHelper (const wxString & ResourceIdentString, HDC PaintDCHandle)
{
    loadBitmap (ResourceIdentString);
    prepare (PaintDCHandle);
};

wxBitmapPaintHelper::wxBitmapPaintHelper (wxBitmap & Bitmap)
{
    setBitmap (Bitmap);
};

wxBitmapPaintHelper::wxBitmapPaintHelper (wxBitmap & Bitmap, HDC PaintDCHandle)
{
    setBitmap (Bitmap);
    prepare (PaintDCHandle);
};

wxBitmapPaintHelper::wxBitmapPaintHelper (int Width, int Height)
{
    createBitmap (Width, Height);
};

wxBitmapPaintHelper::wxBitmapPaintHelper (int Width, int Height, HDC PaintDCHandle)
{
    createBitmap (Width, Height);
    prepare (PaintDCHandle);
};

wxBitmapPaintHelper::~wxBitmapPaintHelper ()
{
    release ();
};

void wxBitmapPaintHelper::setBitmap (wxBitmap & Bitmap)
{
    m_BitmapRef = & Bitmap;
    m_BitmapHandle = (HBITMAP) m_BitmapRef -> GetHBITMAP ();
}

void wxBitmapPaintHelper::loadBitmap (const wxString & ResourceIdentString)
{
    // m_BitmapHandle = ::LoadBitmap (GetModuleHandle(NULL), MAKEINTRESOURCE(ResourceIdent));
    m_Bitmap.LoadFile (ResourceIdentString, wxBITMAP_TYPE_BMP_RESOURCE);
    m_BitmapRef = & m_Bitmap;
    m_BitmapHandle = (HBITMAP) m_BitmapRef -> GetHBITMAP ();
}

void wxBitmapPaintHelper::createBitmap (int Width, int Height)
{
    m_Bitmap.Create (Width, Height);
    m_BitmapRef = & m_Bitmap;
    m_BitmapHandle = (HBITMAP) m_BitmapRef -> GetHBITMAP ();
}

void wxBitmapPaintHelper::prepare (HDC PaintDCHandle)
{
    m_MemoryDCHandle = ::CreateCompatibleDC (PaintDCHandle);
    m_PreviousBitmapHandle = (HBITMAP) ::SelectObject (m_MemoryDCHandle, m_BitmapHandle);
}

void wxBitmapPaintHelper::release ()
{
    ::SelectObject (m_MemoryDCHandle, m_PreviousBitmapHandle);
    ::DeleteDC (m_MemoryDCHandle);
}

int wxBitmapPaintHelper::getWidth ()
{
    return m_BitmapRef -> GetWidth ();
}

int wxBitmapPaintHelper::getHeight ()
{
    return m_BitmapRef -> GetHeight ();
}

#endif // defined(__WXMSW__)

//=========================================================================

#if defined(__WXMSW__)

int getActualUsedColorCount (BITMAPINFO * BitmapInfo)
{
    int ColorsUsed = BitmapInfo -> bmiHeader.biClrUsed;
    // (for BI_RGB):
    // biClrUsed == 0 is interpreted differently depending on the
    // bit depth used:
    // - for color depths from 1..8 bpp: if biClrUsed is set to zero
    //   this means that a palette for all possible colors is there
    // - for color depths 16, 24 or 32 bpp: if biClrUsed is set to 
    //   zero this means that no palette is used (but biClrUsed
    //   may be set to non-zero if a palette is there for 
    //   output optimization purposes)
    if (ColorsUsed == 0 
        && BitmapInfo -> bmiHeader.biBitCount >= 1 
        && BitmapInfo -> bmiHeader.biBitCount <= 8) // FUZZY 8
      ColorsUsed = (1 << BitmapInfo -> bmiHeader.biBitCount);

    return ColorsUsed;
}

#endif // defined(__WXMSW__)

//=========================================================================

void WxBeep (int MilliSeconds)
{
    wxMinimalTimeInterval MinimalTimeInterval;
    MinimalTimeInterval.setMinimalTimeInterval (MilliSeconds);
    
    while (!MinimalTimeInterval.check ())
      {
        wxBell ();
      }
}

//=========================================================================

void doRotateMirrorCoord (int RotationAngle, 
                          bool IsHorizontallyMirrored, 
                          bool IsVerticallyMirrored,
                          const wxPoint & CentrePoint,
                          const wxPoint & InputPoint,
                          wxPoint & OutputPoint,
                          bool IsRotateCenter)
{
    wxPoint DistanceAsPoint = InputPoint - CentrePoint;

    wxPoint RotatedDistanceAsPoint = DistanceAsPoint;
    switch (RotationAngle)
      {
      case 90:
        RotatedDistanceAsPoint.x = DistanceAsPoint.y;
        RotatedDistanceAsPoint.y = -DistanceAsPoint.x;
        break;
      case 180:
        RotatedDistanceAsPoint.x = -DistanceAsPoint.x;
        RotatedDistanceAsPoint.y = -DistanceAsPoint.y;
        break;
      case 270:
        RotatedDistanceAsPoint.x = -DistanceAsPoint.y;
        RotatedDistanceAsPoint.y = DistanceAsPoint.x;
        break;
      }
    
    // PERFORMANCE NOTE doing the negation is possibly faster than 
    // multiplicating with (-1)
    wxPoint MirroredDistanceAsPoint = RotatedDistanceAsPoint;
    if (IsHorizontallyMirrored)
      {
        MirroredDistanceAsPoint.x = -RotatedDistanceAsPoint.x;
      }
    if (IsVerticallyMirrored)
      {
        MirroredDistanceAsPoint.y = -RotatedDistanceAsPoint.y;
      }

    wxPoint NewCentrePoint;
    if (IsRotateCenter
        && (RotationAngle == 90 || RotationAngle == 270))
      {
        NewCentrePoint.x = CentrePoint.y;
        NewCentrePoint.y = CentrePoint.x;
      }
    else
      {
        NewCentrePoint.x = CentrePoint.x;
        NewCentrePoint.y = CentrePoint.y;
      }
    OutputPoint = NewCentrePoint + MirroredDistanceAsPoint;
}

//=========================================================================

wxRotationMirrorState::wxRotationMirrorState()
{
    m_IsHorizontallyMirrored = FALSE;
    m_IsVerticallyMirrored = FALSE;
    m_RotationAngle = 0;

    m_IsUpdateRequired = TRUE;
}

wxRotationMirrorState::~wxRotationMirrorState()
{
}

void wxRotationMirrorState::reset()
{
    m_IsHorizontallyMirrored = FALSE;
    m_IsVerticallyMirrored = FALSE;
    m_RotationAngle = 0;

    m_IsUpdateRequired = TRUE;
}

//-------------------------------------------------------------------------

void wxRotationMirrorState::mirror (bool IsHorizontallyMirrored, bool IsVerticallyMirrored)
{
//  handleMirrorRotation (m_RotationAngle,
//                        IsHorizontallyMirrored,
//                        IsVerticallyMirrored);

    if (IsHorizontallyMirrored)
      {
        m_IsHorizontallyMirrored = !m_IsHorizontallyMirrored;
      }
    if (IsVerticallyMirrored)
      {
        m_IsVerticallyMirrored = !m_IsVerticallyMirrored;
      }

    m_IsUpdateRequired = TRUE;
}

void wxRotationMirrorState::rotate (int RotationAngle)
{
    m_RotationAngle += RotationAngle;
    m_RotationAngle = (((int) m_RotationAngle) / 90) * 90;
    if (m_RotationAngle < 0)
      m_RotationAngle += 360;
    m_RotationAngle %= 360;

    handleMirrorRotation (RotationAngle,
                          m_IsHorizontallyMirrored,
                          m_IsVerticallyMirrored);

    m_IsUpdateRequired = TRUE;
}

void wxRotationMirrorState::handleMirrorRotation(int RotationAngle,
                                                bool & IsHorizontallyMirrored,
                                                bool & IsVerticallyMirrored) 
{
    if (RotationAngle < 0)
      RotationAngle += 360;
    RotationAngle %= 360;
    if (RotationAngle == 90 
        || RotationAngle == 270)
      {
        if ((IsHorizontallyMirrored && !IsVerticallyMirrored)
            || (!IsHorizontallyMirrored && IsVerticallyMirrored))
          {
            // swap mirroring axes
            bool TempVerticallyMirrored = IsVerticallyMirrored;
            IsVerticallyMirrored = IsHorizontallyMirrored;
            IsHorizontallyMirrored = TempVerticallyMirrored;
          }
      }
}

void wxRotationMirrorState::getNormalizedState(int & RotationAngle,
                                              bool & IsHorizontallyMirrored,
                                              bool & IsVerticallyMirrored) 
{
    IsHorizontallyMirrored = m_IsHorizontallyMirrored;
    IsVerticallyMirrored = m_IsVerticallyMirrored;
    RotationAngle = m_RotationAngle;

    // normalize to reduce number of required image transformations:
    if (IsHorizontallyMirrored 
        && IsVerticallyMirrored)
      {
        RotationAngle += 180;
        RotationAngle %= 360;
        IsHorizontallyMirrored = FALSE;
        IsVerticallyMirrored = FALSE;
      }
}

//-------------------------------------------------------------------------

void wxRotationMirrorState::assign (const wxRotationMirrorState & RHSRotationMirrorState)
{
    m_RotationAngle = RHSRotationMirrorState.m_RotationAngle;
    m_IsHorizontallyMirrored = RHSRotationMirrorState.m_IsHorizontallyMirrored;
    m_IsVerticallyMirrored = RHSRotationMirrorState.m_IsVerticallyMirrored;

    m_IsUpdateRequired = TRUE;
}

void wxRotationMirrorState::subtract (const wxRotationMirrorState & LHSRotationMirrorState,
                                      const wxRotationMirrorState & RHSRotationMirrorState)
{
    reset();

    int RotationAngleDifference = LHSRotationMirrorState.m_RotationAngle
      - RHSRotationMirrorState.m_RotationAngle;

    wxRotationMirrorState LHS;
    wxRotationMirrorState RHS;
    LHS.assign (LHSRotationMirrorState);
    RHS.assign (RHSRotationMirrorState);
    
    // rotate LHS to rotation angle of RHS, then determine differences in
    // mirroring, then apply mirroring and finally rotate
    LHS.rotate (- RotationAngleDifference);
    m_IsHorizontallyMirrored = RHS.m_IsHorizontallyMirrored != LHS.m_IsHorizontallyMirrored;
    m_IsVerticallyMirrored = RHS.m_IsVerticallyMirrored != LHS.m_IsVerticallyMirrored;
    rotate (RotationAngleDifference);
    
    m_IsUpdateRequired = TRUE;
}

//=========================================================================

#ifdef M_IsFilteredDIBEnabled

#if defined(__WXMSW__)

wxFilteredDIB::wxFilteredDIB ()
{
    m_Brightness = 0.;
    m_Contrast = 0.;
    m_RedGamma = 1.;
    m_GreenGamma = 1.;
    m_BlueGamma = 1.;
    m_IsInverted = FALSE;
    m_IsColorAdjustmentModified = TRUE;

    // m_IsPaletteBasedColorAdjustmentEnabled = TRUE;
    m_IsOriginalColorPaletteRemembered = FALSE;
    m_IsModifiedColorPaletteRemembered = FALSE;

    m_IsImageMagickFilteringRequired = FALSE;
    m_IsFilteringModified = TRUE;

    m_PreviousRotationMirrorState.reset ();
    m_RotationMirrorState.reset();
    m_RotationMirrorState.m_IsUpdateRequired = TRUE;

    m_SourceDIBHandle = NULL;
    m_IsOwningSourceImage = FALSE;
    m_ResultDIBHandle = NULL;
    m_IsOwningResultImage = TRUE;

    m_FilteredDIBBitsPerPixel = 24;
}

wxFilteredDIB::~wxFilteredDIB ()
{
    freeResultImage ();
    freeSourceImage ();
}

//-------------------------------------------------------------------------

void wxFilteredDIB::setBrightnessContrast (double Brightness, double Contrast)
{
    m_Brightness = Brightness;
    m_Contrast = Contrast;
    m_IsColorAdjustmentModified = TRUE;
}

void wxFilteredDIB::setColorAdjustment (double RedGamma, double GreenGamma,
                                       double BlueGamma)
{
    m_RedGamma = RedGamma;
    m_GreenGamma = GreenGamma;
    m_BlueGamma = BlueGamma;
    m_IsColorAdjustmentModified = TRUE;
}

void wxFilteredDIB::setInversion (bool IsInverted)
{
    m_IsInverted = IsInverted;
    m_IsColorAdjustmentModified = TRUE;
}

//-------------------------------------------------------------------------

bool wxFilteredDIB::rememberPalette (WXHDIB DIBHandle,
                                       wxFilteredDIBColorPaletteArray & ColorPaletteByteArray)
{
    int DIBColorPaletteEntryCount = ::GetActualUsedColorCountInDIBHandle (DIBHandle);

    ColorPaletteByteArray.Clear ();

    if (DIBColorPaletteEntryCount == 0)
      return FALSE;

    ColorPaletteByteArray.Insert ((BYTE) 0, /* array index */ 0,
                                  DIBColorPaletteEntryCount);

    // HACK WARN access of items in m_OriginalColorPaletteByteArray
    ::ImportExportColorPaletteIntoDIB ((BYTE *) (& ColorPaletteByteArray.Item(0)),
                                       DIBColorPaletteEntryCount,
                                       DIBHandle,
                                       FALSE);

    return TRUE;
}

bool wxFilteredDIB::restorePalette (WXHDIB DIBHandle,
                                      wxFilteredDIBColorPaletteArray & ColorPaletteByteArray)
{
    int ColorPaletteEntryCount = ColorPaletteByteArray.GetCount();
    
    if (ColorPaletteEntryCount != 0)
      {
        // HACK WARN access of items in m_OriginalColorPaletteByteArray
        ::ImportExportColorPaletteIntoDIB ((BYTE *) (& ColorPaletteByteArray.Item(0)),
                                           ColorPaletteEntryCount,
                                           DIBHandle,
                                           TRUE);
      }

    return TRUE;
}

void wxFilteredDIB::rememberOriginalPalette ()
{
    rememberPalette (m_SourceDIBHandle,
                     m_OriginalColorPaletteByteArray);
    m_IsOriginalColorPaletteRemembered = TRUE;
}

void wxFilteredDIB::rememberModifiedPalette ()
{
    rememberPalette (m_SourceDIBHandle,
                     m_ModifiedColorPaletteByteArray);
    m_IsModifiedColorPaletteRemembered = TRUE;
}

void wxFilteredDIB::setOriginalPaletteIntoSourceImage ()
{
    if (m_IsOriginalColorPaletteRemembered)
      {
        restorePalette (m_SourceDIBHandle,
                        m_OriginalColorPaletteByteArray);
      }
}

void wxFilteredDIB::setModifiedPaletteIntoSourceImage ()
{
    if (m_IsModifiedColorPaletteRemembered)
      {
        restorePalette (m_SourceDIBHandle,
                        m_ModifiedColorPaletteByteArray);
      }
}

//-------------------------------------------------------------------------

void wxFilteredDIB::resetRotationMirror ()
{
    m_RotationMirrorState.reset();
}

void wxFilteredDIB::rotate (int RotationAngle)
{
    m_RotationMirrorState.rotate (RotationAngle);
}

void wxFilteredDIB::mirror (bool IsHorizontallyMirrored, bool IsVerticallyMirrored)
{
    m_RotationMirrorState.mirror (IsHorizontallyMirrored, IsVerticallyMirrored);
}

//-------------------------------------------------------------------------

void wxFilteredDIB::setIsFilteringModified ()
{
    m_IsFilteringModified = TRUE;
}

void wxFilteredDIB::setIsImageMagickFilteringRequired (bool IsRequired)
{
    m_IsImageMagickFilteringRequired = IsRequired;
    m_IsFilteringModified = TRUE;
}

//-------------------------------------------------------------------------

void wxFilteredDIB::setSourceImage (const WXHDIB & SourceDIBHandle)
{
    m_SourceDIBHandle = SourceDIBHandle;
    m_IsColorAdjustmentModified = TRUE;
    m_IsFilteringModified = TRUE;
}

void wxFilteredDIB::getResultImage (WXHDIB & ResultDIBHandle)
{
    if (m_ResultDIBHandle == NULL)
      ResultDIBHandle = m_SourceDIBHandle;
    else
      ResultDIBHandle = m_ResultDIBHandle;
}

void wxFilteredDIB::setIsOwningSourceImage (bool IsOwningSourceImage)
{
    m_IsOwningSourceImage = IsOwningSourceImage;
}

void wxFilteredDIB::setIsOwningResultImage (bool IsOwningResultImage)
{
    m_IsOwningResultImage = IsOwningResultImage;
}

//-------------------------------------------------------------------------

void wxFilteredDIB::updateResultImage()
{
    if (! (m_IsColorAdjustmentModified
           || m_IsFilteringModified
           || m_RotationMirrorState.m_IsUpdateRequired))
      return;

    // rotation mirror optmization: avoid applying color adjustment and filtering
    // if only rotation or mirroring has changed - this assumes that rotation 
    // and mirroring doesn't change image contents (apart from rotation and 
    // mirroring)
    bool IsRotationMirrorOnly = m_RotationMirrorState.m_IsUpdateRequired
      && ! m_IsColorAdjustmentModified
      && ! m_IsFilteringModified;

    for_once
      {
        if (!IsRotationMirrorOnly)
          freeResultImage ();

        // obtain original DIB
        if (m_SourceDIBHandle == NULL)
            break;

        //-------------------------------------------------------------------------
        // try to apply color adjustment to palette DIBs

        bool IsPaletteBasedColorAdjustmentDone = FALSE;

        if (m_IsPaletteBasedColorAdjustmentEnabled
            && (! IsRotationMirrorOnly))
          {
            // do palette modification using an ImageMagick filter - to this end, 
            // convert to 'image', then run filtering, then write result back
            // into the DIB palette
            ExceptionInfo exceptionInfo;
            GetExceptionInfo (& exceptionInfo);
    
            Image * PaletteMagickImage = NULL;

            if (!m_IsOriginalColorPaletteRemembered)
              {
                rememberOriginalPalette ();
              }
            else
              {
                setOriginalPaletteIntoSourceImage ();
              }

            bool IsPaletteImage = ::DIBPaletteToImage (m_SourceDIBHandle, 
                                                       & PaletteMagickImage,
                                                       & exceptionInfo);

            if (IsPaletteImage)
              {
                if (PaletteMagickImage == NULL)
                  break;

                // note that applyColorAdjustment() is an in-place operation
                applyColorAdjustment (& PaletteMagickImage, & exceptionInfo);
                if (PaletteMagickImage == NULL)
                  break;
            
                bool IsOk = ::ImageToDIBPalette (PaletteMagickImage, 
                                                 m_SourceDIBHandle,
                                                 & exceptionInfo);
                ::DestroyImage (PaletteMagickImage);
                PaletteMagickImage = NULL;

                // WARN should add exception/error handling/parsing after 
                // each function

                if (IsOk == FALSE)
                  break;
                
                IsPaletteBasedColorAdjustmentDone = TRUE;

                rememberModifiedPalette ();
              }
          }

        WXHDIB IntermediateDIBHandle = m_SourceDIBHandle; 
        WXHDIB PreviousIntermediateDIBHandle = m_SourceDIBHandle;
        
        //-------------------------------------------------------------------------
        // convert to ImageMagick image and apply color adjustment and ImageMagick filters

        if ((m_IsImageMagickFilteringRequired
             || !IsPaletteBasedColorAdjustmentDone)
            && (!IsRotationMirrorOnly))
          {
            // do ImageMagick filtering - to this end, convert to 'image'
            // then run filtering, then convert back to DIB
            ExceptionInfo exceptionInfo;
            GetExceptionInfo (& exceptionInfo);
    
            Image * MagickImage = ::DIBToImage (PreviousIntermediateDIBHandle, 
                                                & exceptionInfo);
            if (MagickImage == NULL)
              break;

            // check not to do color adjustment twice
            if (! IsPaletteBasedColorAdjustmentDone)
              {
                applyColorAdjustment (& MagickImage, & exceptionInfo);
              }
    
            applyImageMagickFiltering (& MagickImage, & exceptionInfo);
            if (MagickImage == NULL)
              break;
            
            IntermediateDIBHandle = ::ImageToDIB (MagickImage, 
                                                  m_FilteredDIBBitsPerPixel,
                                                  & exceptionInfo);
            ::DestroyImage (MagickImage);
            MagickImage = NULL;
            if (IntermediateDIBHandle == NULL)
              break;

            // WARN should add exception/error handling/parsing after 
            // each function
            // throwException (exceptionInfo);
            // if (image)
            //    throwException (image->exception);
          }

        // clean up of temporary DIB's
        if (IntermediateDIBHandle != PreviousIntermediateDIBHandle
            && PreviousIntermediateDIBHandle != m_SourceDIBHandle)
          {
            freeImage (PreviousIntermediateDIBHandle);
          }
        PreviousIntermediateDIBHandle = IntermediateDIBHandle;

        //-------------------------------------------------------------------------
        // rotate and mirror:

        int RotationAngle;
        bool IsHorizontallyMirrored;
        bool IsVerticallyMirrored;
        if (IsRotationMirrorOnly)
          {
            wxRotationMirrorState RelativeRotationMirrorState;

            RelativeRotationMirrorState.subtract (m_RotationMirrorState,
                                                  m_PreviousRotationMirrorState);

            RelativeRotationMirrorState.getNormalizedState (RotationAngle, 
                                                            IsHorizontallyMirrored, 
                                                            IsVerticallyMirrored);

            // setup DIB handle for input to rotation function
            if (m_ResultDIBHandle != NULL)
              {
                // if NULL, PreviousIntermediateDIBHandle should be set to
                // m_SourceDIBHandle
                PreviousIntermediateDIBHandle = m_ResultDIBHandle;
              }
          }
        else
          {
            m_RotationMirrorState.getNormalizedState (RotationAngle, 
                                                      IsHorizontallyMirrored, 
                                                      IsVerticallyMirrored);
          }

        if (RotationAngle != 0
            || IsHorizontallyMirrored
            || IsVerticallyMirrored)
          {
            bool IsNewRotatedDIBHandle = FALSE;
            applyImageRotationMirroring (RotationAngle,
                                         IsHorizontallyMirrored,
                                         IsVerticallyMirrored,
                                         PreviousIntermediateDIBHandle,
                                         IntermediateDIBHandle,
                                         IsNewRotatedDIBHandle);

            // release temporary memory
            // if (IsNewFilteredDIBHandle)
            //  freeImage (FilteredDIBHandle);
            if (IntermediateDIBHandle != PreviousIntermediateDIBHandle
                && PreviousIntermediateDIBHandle != m_SourceDIBHandle)
              {
                freeImage (PreviousIntermediateDIBHandle);
              }
            PreviousIntermediateDIBHandle = IntermediateDIBHandle;
          }

        if (PreviousIntermediateDIBHandle != m_SourceDIBHandle)
          {
            m_ResultDIBHandle = PreviousIntermediateDIBHandle;
          }
        else
          {
            m_ResultDIBHandle = NULL;
          }

        m_IsColorAdjustmentModified = FALSE;
        m_IsFilteringModified = FALSE;
        m_RotationMirrorState.m_IsUpdateRequired = FALSE;
        m_PreviousRotationMirrorState.assign (m_RotationMirrorState);

        handleResultImageChanged ();
      }
}

//-------------------------------------------------------------------------

void wxFilteredDIB::handleResultImageChanged ()
{
}

//-------------------------------------------------------------------------

void wxFilteredDIB::applyColorAdjustment (struct _Image ** MagickImagePtrPtr,
                                            struct _ExceptionInfo * exceptionInfo)
{
    for_once
      {
        Image * MagickImage = *MagickImagePtrPtr;
        if (MagickImage == NULL)
          break;

        if (m_IsInverted)
          NegateImage (MagickImage, 0);

        // WARN ContrastImage() seems to use only the sign of 
        // the 'sharpen' parameter
        ContrastImage (MagickImage, (m_Contrast * 100));

        // support for brightness currently done by modifying the gamma
        // of each channel
        double RedGamma = m_Brightness + m_RedGamma;
        double GreenGamma = m_Brightness + m_GreenGamma;
        double BlueGamma = m_Brightness + m_BlueGamma;

        levelGamma (RedGamma);
        levelGamma (GreenGamma);
        levelGamma (BlueGamma);

        GammaImageChannel (MagickImage, RedChannel, RedGamma);
        GammaImageChannel (MagickImage, GreenChannel, GreenGamma);
        GammaImageChannel (MagickImage, BlueChannel, BlueGamma);
      }
}

void wxFilteredDIB::levelGamma (double & Gamma)
{
    if (Gamma < 1.)
      {
        Gamma = 1. - (1. - Gamma) * 0.75;
      }
    else if (Gamma > 1.)
      {
        Gamma = 1. + (Gamma - 1.) * 1.75;
      }
    if (Gamma < 0.1)
      {
        Gamma = 0.1;
      }
}

void wxFilteredDIB::applyImageMagickFiltering (struct _Image ** IMImage,
                                                 struct _ExceptionInfo * exceptionInfo)
{
}

void wxFilteredDIB::applyImageRotationMirroring (int RotationAngle,
                                                   bool IsHorizontallyMirrored,
                                                   bool IsVerticallyMirrored,
                                                   WXHDIB & InputDIBHandle,
                                                   WXHDIB & OutputDIBHandle,
                                                   bool & IsNewOutputDIBHandle)
{
    WXHDIB IntermediateDIBHandle = InputDIBHandle; 
    WXHDIB PreviousIntermediateDIBHandle = InputDIBHandle;

    // rotate image data if required
    switch (RotationAngle)
      {
      case 90:
        IntermediateDIBHandle = applyDIBRotation (PreviousIntermediateDIBHandle, 90);
        break;
      case 180:
        IntermediateDIBHandle = applyDIBRotation (PreviousIntermediateDIBHandle, 180);
        break;
      case 270:
        IntermediateDIBHandle = applyDIBRotation (PreviousIntermediateDIBHandle, 270);
        break;
      }

    // clean up of temporary DIB's
    if (IntermediateDIBHandle != PreviousIntermediateDIBHandle
        && PreviousIntermediateDIBHandle != InputDIBHandle)
      {
        freeImage (PreviousIntermediateDIBHandle);
      }
    PreviousIntermediateDIBHandle = IntermediateDIBHandle;

    // apply horizontal mirroring if required
    if (IsHorizontallyMirrored)
      {
        IntermediateDIBHandle = applyDIBMirroring (PreviousIntermediateDIBHandle, FALSE);
      }

    // clean up of temporary DIB's
    if (IntermediateDIBHandle != PreviousIntermediateDIBHandle
        && PreviousIntermediateDIBHandle != InputDIBHandle)
      {
        freeImage (PreviousIntermediateDIBHandle);
      }
    PreviousIntermediateDIBHandle = IntermediateDIBHandle;

    // apply vertical mirroring if required
    if (IsVerticallyMirrored)
      {
        IntermediateDIBHandle = applyDIBMirroring (PreviousIntermediateDIBHandle, TRUE);
      }

    // clean up of temporary DIB's
    if (IntermediateDIBHandle != PreviousIntermediateDIBHandle
        && PreviousIntermediateDIBHandle != InputDIBHandle)
      {
        freeImage (PreviousIntermediateDIBHandle);
      }
    PreviousIntermediateDIBHandle = IntermediateDIBHandle;

    // set output variables
    OutputDIBHandle = PreviousIntermediateDIBHandle;
    IsNewOutputDIBHandle = InputDIBHandle != OutputDIBHandle;
}

//-------------------------------------------------------------------------

void wxFilteredDIB::freeSourceImage()
{
    if (m_IsOwningSourceImage)
      {
        freeImage (m_SourceDIBHandle);
      }
}

void wxFilteredDIB::freeResultImage()
{
    if (m_IsOwningResultImage)
      {
        freeImage (m_ResultDIBHandle);
      }
}

void wxFilteredDIB::freeImage (WXHDIB & DIBHandle)
{
    if (DIBHandle != NULL)
      {
        GlobalFree (DIBHandle);
        DIBHandle = NULL;
      }
}

#endif // defined(__WXMSW__)

#endif

//=========================================================================

wxChoiceManager::wxChoiceManager ()
{
    m_FocusedSourceItemIndex = -1;
    m_FocusedChoiceItemIndex = -1;
}

void wxChoiceManager::OnSelect ()
{
    selectIntoChoice (getChoiceFocus(), getSourceFocus());
}

void wxChoiceManager::OnDeselect ()
{
    deselectFromChoice (getChoiceFocus());
}

void wxChoiceManager::OnClearSelection ()
{
    int ChoiceItemCount = getChoiceItemCount ();
    for (int ChoiceItemIndex=0; ChoiceItemIndex<ChoiceItemCount; ++ChoiceItemIndex)
      {
        deselectFromChoice (ChoiceItemIndex, FALSE);
      }
    updateChoiceView ();
}

void wxChoiceManager::OnSourceLeftMouseButton (int SourceItemIndex)
{
    bool IsNewlyFocused = getSourceFocus () != SourceItemIndex;
        
    if (IsNewlyFocused)
      setSourceFocus (SourceItemIndex);

    handleSelection (IsNewlyFocused, getChoiceFocus (), SourceItemIndex);
}

void wxChoiceManager::OnSourceRightMouseButton (int SourceItemIndex)
{
    int DocumentIdent = getSourceDocument (SourceItemIndex);
    deselectFromChoice (findChoiceItemIndex (DocumentIdent));
}

void wxChoiceManager::OnChoiceLeftMouseButton (int ChoiceItemIndex)
{
    bool IsNewlyFocused = getChoiceFocus () != ChoiceItemIndex;

    if (IsNewlyFocused)
      setChoiceFocus (ChoiceItemIndex);

    handleSelection (IsNewlyFocused, ChoiceItemIndex, getSourceFocus());
}

void wxChoiceManager::OnChoiceRightMouseButton (int ChoiceItemIndex)
{
    deselectFromChoice (ChoiceItemIndex);
}

//-------------------------------------------------------------------------

void wxChoiceManager::handleSelection (bool IsNewlyFocused, int ChoiceItemIndex,
                                         int SourceItemIndex)
{
    bool IsChoiceHavingDocument =  getChoiceDocument (ChoiceItemIndex) != -1;
    bool IsDocumentAlreadyAssigned = findChoiceItemIndex (getSourceDocument (SourceItemIndex)) != -1;
    
    if ((!IsNewlyFocused || (!IsChoiceHavingDocument && !IsDocumentAlreadyAssigned))
        && SourceItemIndex != -1 
        && ChoiceItemIndex != -1)
      {
        selectIntoChoice (ChoiceItemIndex, SourceItemIndex);
      }
}

//-------------------------------------------------------------------------

void wxChoiceManager::selectIntoChoice (int ChoiceItemIndex, int SourceItemIndex,
                                                 bool IsUpdateChoiceView)
{
    int ChoiceItemCount = getChoiceItemCount();

    if (ChoiceItemIndex != -1 && SourceItemIndex != -1)
      {
        // create entries of -1 if necessary
        if (ChoiceItemIndex >= ChoiceItemCount)
          {
            insertChoiceDocument (-1, ChoiceItemCount, ChoiceItemIndex + 1 - ChoiceItemCount);
          }

        deselectFromChoice (ChoiceItemIndex, 
                             /* IsUpdateChoiceView */ FALSE);

        setSelection (SourceItemIndex, TRUE);
        int DocumentIdent = getSourceDocument (SourceItemIndex);
        setChoiceDocument (ChoiceItemIndex, DocumentIdent);

        if (IsUpdateChoiceView)
          updateChoiceView ();
      }
}

void wxChoiceManager::deselectFromChoice (int ChoiceItemIndex, 
                                                   bool IsUpdateChoiceView)
{
    if (ChoiceItemIndex != -1 && ChoiceItemIndex < getChoiceItemCount())
      {
        int DocumentIdent = getChoiceDocument (ChoiceItemIndex);

        if (DocumentIdent != -1)
          {
            int SourceItemIndex = findSourceItemIndex (DocumentIdent);
            setChoiceDocument (ChoiceItemIndex, -1);
            int OtherChoiceItemIndex = findChoiceItemIndex (DocumentIdent);
            setSelection (SourceItemIndex, OtherChoiceItemIndex != -1);

            if (IsUpdateChoiceView)
              updateChoiceView ();
          }
      }
}

//-------------------------------------------------------------------------

int wxChoiceManager::getSourceFocus ()
{
    return m_FocusedSourceItemIndex;
}

void wxChoiceManager::setSourceFocus (int NewFocusedItemIndex)
{
    if (NewFocusedItemIndex == m_FocusedSourceItemIndex)
      return;

    if (m_FocusedSourceItemIndex != -1)
      {
        updateSourceFocus (m_FocusedSourceItemIndex, FALSE);
      }
    m_FocusedSourceItemIndex = NewFocusedItemIndex;

    if (m_FocusedSourceItemIndex != -1)
      {
        updateSourceFocus (m_FocusedSourceItemIndex, TRUE);
      }
}

int wxChoiceManager::getChoiceFocus ()
{
    return m_FocusedChoiceItemIndex;
}

void wxChoiceManager::setChoiceFocus (int NewFocusedItemIndex)
{
    if (NewFocusedItemIndex == m_FocusedChoiceItemIndex)
      return;

    if (m_FocusedChoiceItemIndex != -1)
      {
        updateChoiceFocus (m_FocusedChoiceItemIndex, FALSE);
      }
    m_FocusedChoiceItemIndex = NewFocusedItemIndex;

    if (m_FocusedChoiceItemIndex != -1)
      {
        updateChoiceFocus (m_FocusedChoiceItemIndex, TRUE);
      }
}

void wxChoiceManager::redoSelections ()
{
    int SourceItemCount = getSourceItemCount ();
    for (int SourceItemIndex=0; SourceItemIndex<SourceItemCount; ++SourceItemIndex)
      {
        int DocumentIdent = getSourceDocument (SourceItemIndex);
        setSelection (SourceItemIndex, findChoiceItemIndex (DocumentIdent) != -1);
      }
    updateChoiceView ();
}

void wxChoiceManager::setSelection (int SourceItemIndex, bool IsSelect)
{
    if (SourceItemIndex == -1)
      return;

    updateSelection (SourceItemIndex, IsSelect);
}

//-------------------------------------------------------------------------

int wxChoiceManager::findSourceItemIndex (int DocumentIdent)
{
    int SourceDocumentCount = getSourceItemCount();

    for (int SourceDocumentIndex=0; SourceDocumentIndex<SourceDocumentCount; ++SourceDocumentIndex)
      {
        if (getSourceDocument (SourceDocumentIndex) == DocumentIdent)
          return SourceDocumentIndex;
      }

    return -1;
}

int wxChoiceManager::findChoiceItemIndex (int DocumentIdent)
{
    int ChoiceDocumentCount = getChoiceItemCount ();

    for (int ChoiceDocumentIndex=0; ChoiceDocumentIndex<ChoiceDocumentCount; ++ChoiceDocumentIndex)
      {
        if (getChoiceDocument (ChoiceDocumentIndex) == DocumentIdent)
          return ChoiceDocumentIndex;
      }

    return -1;
}

//=========================================================================
