
/*
-------------------------------------------------------------------------
This file is part of WxWidgetsExtensions library.
-------------------------------------------------------------------------

WxWidgetsExtensions library 0.7.1
-----------------------------

COPYRIGHT NOTICE:

WxWidgetsExtensions library Copyright (c) 2003, 2004 Daniel Kps.

The WxWidgetsExtensions library and associated documentation files (the
"Software") is provided "AS IS".  The author(s) disclaim all
warranties, expressed or implied, including, without limitation, the
warranties of merchantability and of fitness for any purpose.  The
author(s) assume no liability for direct, indirect, incidental,
special, exemplary, or consequential damages, which may result from
the use of or other dealings in the Software, even if advised of the
possibility of such damage.

Permission is hereby granted, free of charge, to any person obtaining
a copy of this Software, to deal in the Software without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Software,
and to permit persons to whom the Software is furnished to do so,
subject to the following conditions:

 1. The origin of this source code must not be misrepresented.
 2. Altered versions must be plainly marked as such and must not be
    misrepresented as being the original source.
 3. This Copyright notice may not be removed or altered from any 
    source or altered source distribution.

End of WxWidgetsExtensions library Copyright notice

-------------------------------------------------------------------------

RATIONALE

The normal wxMessageBox supports only predefined sets of Buttons --
this is a disadvantage, because the naming of the buttons cannot be
done specific to the problem. The used messagebox implementations in
X11, Motif, GTK, MSW seem to be equally restricted.

Examples (flags for MSW C-API): 
1) An error occured, but there is only the MB_OK style (which produces a
   MessageBox with a single 'OK' button). However, a single 'Cancel'
   button would be more appropriate.
2) The user is asked whether he/she wants to save a file. The buttons
   can be named 'Yes/No/Cancel' with the MB_YESNOCANCEL style, but with
   the normal MessageBox we cannot label the buttons
   'Save/Discard/Cancel'. 'Save/Discard/Cancel' should be faster
   recognizable than 'Yes/No/Cancel', because the actions are quite clear
   even without reading the question.

Another disadvantage of the normal MessageBox implementations is that
the message text cannot be copied to the clipboard. This is just
impractical if the user wants, for example, to note an uncommon error
message. Moreover, automatic linebreak and automatic use of scrollbars
for long message texts would be desireable.

wxMessageExtDialog tries to overcome these shortcomings by providing a
dialog class which allows specification of a message text, a title and
arbitrary named buttons. It also tries to keep the usage of the
programming interface simple, so that using the class isn't much
more complicated than calling e.g. wxMessageBox().

Automatic line-break (wrapping) and scrolling is solved in
wxMessageExtDialog by using either a wxTextCtrl or wxHtmlWindow (in case
wxTextCtrl's functionality is not sufficient under the used platform).

Copying to the clipboard is possible if the used control supports it
(works for wxTextCtrl under MSW; doesn't work for wxHtmlWindow in
wxWindows 2.4.1; works for wxHtmlWindow in wxWidgets 2.5.2 in GTK
port).

------------------------------------------------------------------------- */

#ifndef _INCLUDED_MessageExtDialog_h
#define _INCLUDED_MessageExtDialog_h

#define M_MessageExtDialog_EnableCInterface
#ifdef __cplusplus
#   define M_MessageExtDialog_EnableCxxInterface
#endif

// can #define M_MessageExtDialog_EnableCInterface to get 
// global/static functions to display a dialog box callable from C-code.
// May be useful if an application consists of mixed C and C++ source code
// (these C functions are only wrappers, actual functionality is provided 
// from C++ code).

#ifdef M_MessageExtDialog_EnableCxxInterface

#if defined(__GNUG__) && (!defined(__APPLE__)) && (!(defined M_NoPragmaInterface))
#   pragma interface "MessageExtDialog.h"
#endif

#ifndef WX_PRECOMP
#   include <wx/dialog.h>
#endif

#include "WxExtDialog.h"
#include "WxMisc.h"

//-------------------------------------------------------------------------

// wxMessageExtDialog:
// - support for arbitrary named buttons
// - support for automatic line-wrap and use of scrollbars in case
//   of long message texts
// - indication of message box style via wxColourBarWindow
// - message and button text font is a little bigger than normal text
//   to increase readability
// - allows definition of a global title postfix added to all wxMessageExtDialog
//   titles (could be set, for example, to the application's name)
// - easy to use functions MessageExtDialog() (in case only an 'OK' button
//   is required)
// - finally, a HTML mode was added: it is now possible to display HTML messages
// Notes:
// - don't confuse this class with wxMessageDialog - wxMessageDialog is currently
//   only a wrapper to the platform specific message box implementation;
//   wxGenericMessageDialog is also quite restricted
class wxMessageExtDialog: public wxExtDialog
{
public:
    // DEF if the IsHtmlMode flag is specified, MessageString must be a HTML
    // string (without the <html>, <body> tags - they are added automatically)

    //-------------------------------------------------------------------------
    //-- construction, destruction
    wxMessageExtDialog ();
    wxMessageExtDialog (wxWindow *ParentWindow,
                      const wxString & MessageString,
                      const wxString & TitleString = wxEmptyString);
    void Create (wxWindow *ParentWindow,
                 const wxString & MessageString,
                 const wxString & TitleString = wxEmptyString);
    void init (const wxString & MessageString, const wxString & TitleString);

    //-------------------------------------------------------------------------
    //-- set title postfix that is added to all titles
    static void setGlobalTitlePostfix (const wxString & String);
    static void setGlobalMessageExtDialogZoom (double ZoomValue);
    static double getGlobalMessageExtDialogZoom ();

    //-- message to display
    void setDialogMessage (const wxString & MessageString);

    //-- button managment
    wxMessageExtDialog & addButton (const wxString & Label,
                                  int ButtonIdent);
    // DEF addButton(Label) returns the generated ButtonIdent/Control Id
    int addButton (const wxString & Label);
    void makeLastAddedButtonDefault ();

    wxButton * getButton (int ButtonIdent);
    void enableButton (int ButtonIdent, bool IsEnable = TRUE);
    void setButtonLabel (int ButtonIdent, const wxString & LabelString);

    //-- status control
    // (status control could be useful if deriving from wxMessageExtDialog)
    wxMessageExtDialog & addStatusTextCtrl ();

    //-- message box style
    // style is visualized with a wxColourBarWindow (don't like icons
    // to visualize message box type, and believe that for the purpose
    // of visualizing message box type, an icon does not make a lot of 
    // sense from a perceptual point of view - maybe unless the icon/image
    // is specific for the problem/message box text).
    enum EStyle 
    {
      MessageStyle          = 0x0001, 
      InformationStyle,
      WarningStyle,
      ErrorStyle,
      CriticalErrorStyle,
      QuestionStyle,
      StyleMask             = 0x000F,
      
      IsHtmlMode            = 0x0100
    };
    /* EStyle */ long m_Style;
    void setStyle (/* EStyle */ long Style);

    //-- show dialog and return pressed ButtonIdent
    virtual int ShowModal ();

    //-------------------------------------------------------------------------
    //-- message handling:
    virtual void OnCancel(wxCommandEvent & Event);
    virtual void OnOK(wxCommandEvent & Event);
    virtual void OnButtonEvent (wxCommandEvent & Event);
    virtual void handleButtonEvent (wxCommandEvent & Event);

    void OnChar (wxKeyEvent & KeyEvent);
    void OnKeyDown (wxKeyEvent & KeyEvent);

    //-- actual creation of controls
    void createControls ();

 private:

    //-------------------------------------------------------------------------
    wxFont m_TextCtrlFont;
    wxFont m_ButtonFont;

    wxSizer * m_PanelSizer;
    wxMultiLineText m_MessageMultiLineText;

    wxFlexGridSizer * m_ButtonFlexGridSizer;
    wxButton * m_DefaultButton;
    wxButton * m_LastAddedButton;

    //-------------------------------------------------------------------------
    // WARN Automatically choosen button idents may conflict with dialog
    // controls in other windows, must check this
    // maybe one should better use the automatically choosen control idents 
    // choosen by wxWindows, don't know which is better
    enum
    {
      FirstAutoButtonIdent = 50
    };
    int m_LastAddedAutoButtonIdent;
    int m_CurrentButtonIndex;

    wxString m_DialogMessageString;
    wxString m_TitleString;
    static wxString m_GlobalTitlePostfix;
    static double m_GlobalMessageExtDialogZoom;

    bool m_IsControlsCreated;

 public:
    wxButton * findButtonForKey (char KeyChar);

    DECLARE_EVENT_TABLE()
};


//-------------------------------------------------------------------------

// easy-to-use functions to display a message dialog that contains
// the specified message and an "OK" button
// - if the IsHtmlMode flag is specified, MessageString must be a HTML
//   string (without the <html>, <body> tags - they are added automatically)
// TODO 
// - add possibility to specify button label
// - could provide overloads for one, two, three (arbitrary) buttons
int MessageExtDialog (wxWindow * ParentWindow,
                     const wxString & MessageString,
                     const wxString & TitleString = wxEmptyString,
                     /* wxMessageExtDialog::EStyle */ long Style = wxMessageExtDialog::MessageStyle);

int MessageExtDialog (const wxString & MessageString,
                     const wxString & TitleString = wxEmptyString,
                     /* wxMessageExtDialog::EStyle */ long Style = wxMessageExtDialog::MessageStyle);

int MessageExtDialog (const wxString & MessageString,
                     /* wxMessageExtDialog::EStyle */ long Style);

//-------------------------------------------------------------------------

#endif // M_MessageExtDialog_EnableCxxInterface
#ifdef M_MessageExtDialog_EnableCInterface

// interface for code compiled in C mode:

#ifdef __cplusplus
extern "C"
{
#endif

// repeat style definitions from inside the wxMessageExtDialog class:
enum EMessageExtDialogStyle
{
  wxMessageExtDialog_MessageStyle          = 0x0001, 
  wxMessageExtDialog_InformationStyle,
  wxMessageExtDialog_WarningStyle,
  wxMessageExtDialog_ErrorStyle,
  wxMessageExtDialog_CriticalErrorStyle,
  wxMessageExtDialog_QuestionStyle,
  wxMessageExtDialog_IsHtmlMode            = 0x0100
};

int MessageExtDialog_ (const char * MessageString,
                      EMessageExtDialogStyle Style);

#ifdef __cplusplus
}
#endif

#endif  // M_MessageExtDialog_EnableCInterface

//-------------------------------------------------------------------------

#endif
